; ============================================================================
; Jxon.ahk -- JSON parser and serializer for AutoHotkey v2.0.21+
; ============================================================================
;
; Provides two public functions:
;   Jxon_Load(&str)           -- Parse a JSON string into AHK Map/Array objects.
;   Jxon_Dump(obj, indent)    -- Serialize an AHK Map/Array/value to a JSON string.
;
; Based on the JXON community library (TheArkive/JXON_ahk2), rewritten for
; clarity and strict AHK v2 compliance.
;
; Data type mapping:
;   JSON object   <-->  AHK Map
;   JSON array    <-->  AHK Array
;   JSON string   <-->  AHK String
;   JSON number   <-->  AHK Integer or Float
;   JSON true     <-->  AHK 1 (Integer)
;   JSON false    <-->  AHK 0 (Integer)
;   JSON null     <-->  AHK ""  (empty string)
;
; Usage:
;   ; Parse JSON
;   jsonStr := '{"name":"Claude","version":1,"active":true,"tags":["ahk","ai"]}'
;   obj := Jxon_Load(&jsonStr)
;   MsgBox(obj["name"])           ; "Claude"
;   MsgBox(obj["tags"][1])        ; "ahk"
;
;   ; Serialize to JSON
;   data := Map("greeting", "hello", "count", 42, "items", [1, 2, 3])
;   jsonOut := Jxon_Dump(data)           ; compact
;   jsonPretty := Jxon_Dump(data, 2)     ; indented with 2 spaces
;
; ============================================================================

#Requires AutoHotkey v2.0


; ============================================================================
; Jxon_Load -- Parse a JSON string into native AHK objects.
;
; Parameters:
;   &src  (ByRef String) -- The JSON text to parse. The variable is consumed
;                           (trimmed) during parsing; on return it contains
;                           any remaining unparsed text.
;
; Returns:
;   Map, Array, String, Number, or "" (for null)
;
; Throws:
;   Error on malformed JSON.
; ============================================================================
Jxon_Load(&src) {
    ; Trim leading whitespace
    src := LTrim(src, " `t`n`r")

    if (src = "")
        throw Error("Jxon_Load: empty input", -1)

    return _Jxon_Parse(&src)
}


; ============================================================================
; Internal recursive parser
; ============================================================================
_Jxon_Parse(&src) {
    src := LTrim(src, " `t`n`r")

    if (src = "")
        throw Error("Jxon_Load: unexpected end of input", -1)

    char := SubStr(src, 1, 1)

    ; --- Object ---
    if (char = "{") {
        obj := Map()
        obj.CaseSense := false
        src := LTrim(SubStr(src, 2), " `t`n`r")

        if (SubStr(src, 1, 1) = "}") {
            src := SubStr(src, 2)
            return obj
        }

        loop {
            ; Parse key (must be a string)
            key := _Jxon_Parse(&src)
            if !(key is String)
                throw Error("Jxon_Load: expected string key in object, got " . Type(key), -1)

            ; Expect colon
            src := LTrim(src, " `t`n`r")
            if (SubStr(src, 1, 1) != ":")
                throw Error("Jxon_Load: expected ':' after object key '" . key . "'", -1)
            src := LTrim(SubStr(src, 2), " `t`n`r")

            ; Parse value
            val := _Jxon_Parse(&src)
            obj[key] := val

            ; Expect comma or closing brace
            src := LTrim(src, " `t`n`r")
            nextChar := SubStr(src, 1, 1)
            if (nextChar = "}") {
                src := SubStr(src, 2)
                return obj
            }
            if (nextChar != ",")
                throw Error("Jxon_Load: expected ',' or '}' in object, got '" . nextChar . "'", -1)
            src := LTrim(SubStr(src, 2), " `t`n`r")
        }
    }

    ; --- Array ---
    if (char = "[") {
        arr := []
        src := LTrim(SubStr(src, 2), " `t`n`r")

        if (SubStr(src, 1, 1) = "]") {
            src := SubStr(src, 2)
            return arr
        }

        loop {
            val := _Jxon_Parse(&src)
            arr.Push(val)

            src := LTrim(src, " `t`n`r")
            nextChar := SubStr(src, 1, 1)
            if (nextChar = "]") {
                src := SubStr(src, 2)
                return arr
            }
            if (nextChar != ",")
                throw Error("Jxon_Load: expected ',' or ']' in array, got '" . nextChar . "'", -1)
            src := LTrim(SubStr(src, 2), " `t`n`r")
        }
    }

    ; --- String ---
    if (char = "`"") {
        return _Jxon_ParseString(&src)
    }

    ; --- Number ---
    if (char = "-" || (char >= "0" && char <= "9")) {
        return _Jxon_ParseNumber(&src)
    }

    ; --- true ---
    if (SubStr(src, 1, 4) = "true") {
        src := SubStr(src, 5)
        return 1
    }

    ; --- false ---
    if (SubStr(src, 1, 5) = "false") {
        src := SubStr(src, 6)
        return 0
    }

    ; --- null ---
    if (SubStr(src, 1, 4) = "null") {
        src := SubStr(src, 5)
        return ""
    }

    throw Error("Jxon_Load: unexpected character '" . char . "'", -1)
}


; ============================================================================
; Parse a JSON string token (handles escape sequences)
; ============================================================================
_Jxon_ParseString(&src) {
    if (SubStr(src, 1, 1) != "`"")
        throw Error("Jxon_Load: expected '`"' at start of string", -1)

    result := ""
    i := 2  ; start after opening quote
    len := StrLen(src)

    loop {
        if (i > len)
            throw Error("Jxon_Load: unterminated string", -1)

        ch := SubStr(src, i, 1)

        if (ch = "`"") {
            ; End of string
            src := SubStr(src, i + 1)
            return result
        }

        if (ch = "\") {
            ; Escape sequence
            i += 1
            if (i > len)
                throw Error("Jxon_Load: unterminated escape in string", -1)

            esc := SubStr(src, i, 1)
            switch esc {
                case "`"":  result .= "`""
                case "\":   result .= "\"
                case "/":   result .= "/"
                case "b":   result .= "`b"
                case "f":   result .= "`f"
                case "n":   result .= "`n"
                case "r":   result .= "`r"
                case "t":   result .= "`t"
                case "u":
                    ; Unicode escape: \uXXXX
                    if (i + 4 > len)
                        throw Error("Jxon_Load: incomplete unicode escape", -1)
                    hex := SubStr(src, i + 1, 4)
                    if !RegExMatch(hex, "^[0-9a-fA-F]{4}$")
                        throw Error("Jxon_Load: invalid unicode escape '\u" . hex . "'", -1)
                    codePoint := Integer("0x" . hex)

                    ; Handle UTF-16 surrogate pairs (\uD800-\uDBFF followed by \uDC00-\uDFFF)
                    if (codePoint >= 0xD800 && codePoint <= 0xDBFF) {
                        ; High surrogate -- expect low surrogate
                        if (SubStr(src, i + 5, 2) != "\u")
                            throw Error("Jxon_Load: expected low surrogate after high surrogate", -1)
                        hex2 := SubStr(src, i + 7, 4)
                        if !RegExMatch(hex2, "^[0-9a-fA-F]{4}$")
                            throw Error("Jxon_Load: invalid low surrogate '\u" . hex2 . "'", -1)
                        lowSurrogate := Integer("0x" . hex2)
                        if (lowSurrogate < 0xDC00 || lowSurrogate > 0xDFFF)
                            throw Error("Jxon_Load: invalid low surrogate value", -1)
                        codePoint := 0x10000 + ((codePoint - 0xD800) << 10) + (lowSurrogate - 0xDC00)
                        i += 6  ; skip \uXXXX of low surrogate
                    }

                    result .= Chr(codePoint)
                    i += 4  ; skip the 4 hex digits
                default:
                    throw Error("Jxon_Load: invalid escape character '\" . esc . "'", -1)
            }
            i += 1
        } else {
            result .= ch
            i += 1
        }
    }
}


; ============================================================================
; Parse a JSON number token (integer or float)
; ============================================================================
_Jxon_ParseNumber(&src) {
    ; Match the full JSON number grammar:
    ;   -? (0 | [1-9][0-9]*) (.[0-9]+)? ([eE][+-]?[0-9]+)?
    if !RegExMatch(src, "^(-?(?:0|[1-9]\d*)(?:\.\d+)?(?:[eE][+\-]?\d+)?)", &match)
        throw Error("Jxon_Load: invalid number at '" . SubStr(src, 1, 20) . "'", -1)

    numStr := match[1]
    src := SubStr(src, StrLen(numStr) + 1)

    ; Determine if integer or float
    if InStr(numStr, ".") || InStr(numStr, "e") || InStr(numStr, "E")
        return Float(numStr)
    else
        return Integer(numStr)
}


; ============================================================================
; Jxon_Dump -- Serialize an AHK value to a JSON string.
;
; Parameters:
;   obj     -- The value to serialize (Map, Array, String, Number).
;   indent  -- (Optional) Number of spaces per indentation level for pretty
;              printing. 0 or omitted produces compact JSON.
;
; Returns:
;   String -- The JSON text.
;
; Throws:
;   Error on unsupported types.
;
; Usage:
;   json := Jxon_Dump(Map("key", "value"))       ; '{"key":"value"}'
;   json := Jxon_Dump(Map("key", "value"), 4)     ; pretty-printed
; ============================================================================
Jxon_Dump(obj, indent := 0) {
    return _Jxon_Serialize(obj, indent, 0)
}


; ============================================================================
; Internal recursive serializer
;
; Parameters:
;   obj          -- Value to serialize
;   indentSize   -- Spaces per level (0 = compact)
;   currentDepth -- Current nesting depth
; ============================================================================
_Jxon_Serialize(obj, indentSize, currentDepth) {
    ; --- Null (empty string represents JSON null) ---
    ; We need a way to distinguish "" from null. By convention, we treat
    ; the special unset case as well. For serialization, "" maps to "".
    ; To output null, use the Jxon_Null class below.

    if IsObject(obj) {
        ; Check for Jxon_Null sentinel
        if (obj is Jxon_Null)
            return "null"

        ; --- Map (JSON object) ---
        if (obj is Map) {
            if (obj.Count = 0)
                return "{}"

            items := []
            newDepth := currentDepth + 1

            for key, val in obj {
                keyStr := _Jxon_SerializeString(String(key))
                valStr := _Jxon_Serialize(val, indentSize, newDepth)
                items.Push(keyStr . ":" . (indentSize > 0 ? " " : "") . valStr)
            }

            if (indentSize > 0) {
                innerIndent := _Jxon_Indent(indentSize, newDepth)
                outerIndent := _Jxon_Indent(indentSize, currentDepth)
                return "{`n" . innerIndent . _Jxon_Join(items, ",`n" . innerIndent) . "`n" . outerIndent . "}"
            }
            return "{" . _Jxon_Join(items, ",") . "}"
        }

        ; --- Array (JSON array) ---
        if (obj is Array) {
            if (obj.Length = 0)
                return "[]"

            items := []
            newDepth := currentDepth + 1

            for val in obj {
                items.Push(_Jxon_Serialize(val, indentSize, newDepth))
            }

            if (indentSize > 0) {
                innerIndent := _Jxon_Indent(indentSize, newDepth)
                outerIndent := _Jxon_Indent(indentSize, currentDepth)
                return "[`n" . innerIndent . _Jxon_Join(items, ",`n" . innerIndent) . "`n" . outerIndent . "]"
            }
            return "[" . _Jxon_Join(items, ",") . "]"
        }

        ; Unsupported object type
        throw Error("Jxon_Dump: cannot serialize object of type '" . Type(obj) . "'", -1)
    }

    ; --- String ---
    if (obj is String)
        return _Jxon_SerializeString(obj)

    ; --- Integer ---
    if (obj is Integer)
        return String(obj)

    ; --- Float ---
    if (obj is Float) {
        s := String(obj)
        ; Ensure there is a decimal point so JSON parsers treat it as float
        if !InStr(s, ".") && !InStr(s, "e") && !InStr(s, "E")
            s .= ".0"
        return s
    }

    ; Fallback: convert to string
    return _Jxon_SerializeString(String(obj))
}


; ============================================================================
; Serialize a string with JSON escaping
; ============================================================================
_Jxon_SerializeString(str) {
    ; Escape special characters
    result := "`""
    i := 1
    len := StrLen(str)

    loop {
        if (i > len)
            break

        ch := SubStr(str, i, 1)
        code := Ord(ch)

        switch ch {
            case "`"":     result .= "\`""
            case "\":      result .= "\\"
            case "`n":     result .= "\n"
            case "`r":     result .= "\r"
            case "`t":     result .= "\t"
            case "`b":     result .= "\b"
            case "`f":     result .= "\f"
            default:
                if (code < 0x20) {
                    ; Control character: escape as \u00XX
                    result .= "\u" . Format("{:04x}", code)
                } else {
                    result .= ch
                }
        }
        i += 1
    }

    result .= "`""
    return result
}


; ============================================================================
; Helper: Build indentation string
; ============================================================================
_Jxon_Indent(size, depth) {
    if (size <= 0 || depth <= 0)
        return ""

    total := size * depth
    result := ""
    loop total
        result .= " "
    return result
}


; ============================================================================
; Helper: Join array of strings with a separator
; ============================================================================
_Jxon_Join(arr, sep) {
    result := ""
    for i, item in arr {
        if (i > 1)
            result .= sep
        result .= item
    }
    return result
}


; ============================================================================
; Sentinel class for explicit JSON null values.
;
; Usage:
;   obj := Map("key", Jxon_Null())
;   Jxon_Dump(obj)  ; => {"key":null}
;
;   ; When loading JSON, null becomes "" (empty string).
;   ; If you need to distinguish null from empty string, check for "".
; ============================================================================
class Jxon_Null {
    ToString() {
        return "null"
    }
}


; ============================================================================
; Convenience: Jxon_True and Jxon_False are simply 1 and 0 in AHK v2.
; These aliases exist for readability in code that builds JSON structures.
; ============================================================================
Jxon_True() {
    return 1
}

Jxon_False() {
    return 0
}
