;; UIACapture.ahk -- UI Automation accessibility tree capture module
;; Module 4 of Claude AHK Client
;; Uses the Windows COM-based UI Automation API to capture accessibility
;; tree information about UI elements, providing rich semantic context
;; for Claude to generate robust AHK v2 scripts.
;;
;; Requires: AHK v2.0.21+, Windows 10/11 64-bit
;; License: MIT

#Requires AutoHotkey v2.0

class UIACapture {

    ; --- Properties ---
    uia         := ""       ; IUIAutomation COM interface
    maxDepth    := 5        ; Maximum tree traversal depth
    maxElements := 500      ; Maximum elements per snapshot
    timeout     := 2000     ; Capture timeout in ms
    enabled     := true     ; Whether UIA capture is active
    _isAvailable := false   ; Whether COM object was created successfully
    _elementCount := 0      ; Counter for current tree walk (reset per capture)
    _startTick  := 0        ; Tick count at start of current capture (timeout tracking)

    ; --- UIA Control Type ID Constants ---
    ; Reference: https://learn.microsoft.com/en-us/windows/win32/winauto/uiauto-controltype-ids
    static ControlTypes := Map(
        50000, "Button",
        50001, "Calendar",
        50002, "CheckBox",
        50003, "ComboBox",
        50004, "Edit",
        50005, "Hyperlink",
        50006, "Image",
        50007, "ListItem",
        50008, "List",
        50009, "Menu",
        50010, "MenuBar",
        50011, "MenuItem",
        50012, "ProgressBar",
        50013, "RadioButton",
        50014, "ScrollBar",
        50015, "Slider",
        50016, "Spinner",
        50017, "StatusBar",
        50018, "Tab",
        50019, "TabItem",
        50020, "Text",
        50021, "ToolBar",
        50022, "ToolTip",
        50023, "Tree",
        50024, "TreeItem",
        50025, "Custom",
        50026, "Group",
        50027, "Thumb",
        50028, "DataGrid",
        50029, "DataItem",
        50030, "Document",
        50031, "SplitButton",
        50032, "Window",
        50033, "Pane",
        50034, "Header",
        50035, "HeaderItem",
        50036, "Table",
        50037, "TitleBar",
        50038, "Separator",
        50039, "SemanticZoom",
        50040, "AppBar"
    )

    ; --- UIA Pattern IDs for supported pattern detection ---
    static PatternIds := Map(
        10000, "InvokePattern",
        10001, "SelectionPattern",
        10002, "ValuePattern",
        10003, "RangeValuePattern",
        10004, "ScrollPattern",
        10005, "ExpandCollapsePattern",
        10006, "GridPattern",
        10007, "GridItemPattern",
        10008, "MultipleViewPattern",
        10009, "WindowPattern",
        10010, "SelectionItemPattern",
        10011, "DockPattern",
        10012, "TablePattern",
        10013, "TableItemPattern",
        10014, "TextPattern",
        10015, "TogglePattern",
        10016, "TransformPattern",
        10017, "ScrollItemPattern",
        10018, "LegacyIAccessiblePattern"
    )

    ; =========================================================================
    ; Constructor
    ; =========================================================================

    /**
     * Creates the UIAutomation COM object and configures capture settings.
     * Gracefully handles COM creation failure (some locked-down PCs block UIA).
     *
     * @param {Map} config - Optional configuration Map with keys:
     *   "maxDepth"    (Integer) - Max tree traversal depth
     *   "maxElements" (Integer) - Max elements per snapshot
     *   "timeout"     (Integer) - Capture timeout in ms
     *   "enabled"     (Boolean) - Whether UIA capture is active
     */
    __New(config := unset) {
        ; Apply config overrides if provided
        if IsSet(config) && IsObject(config) {
            if config.Has("maxDepth") && IsInteger(config["maxDepth"])
                this.maxDepth := Max(1, Min(config["maxDepth"], 20))
            if config.Has("maxElements") && IsInteger(config["maxElements"])
                this.maxElements := Max(10, Min(config["maxElements"], 5000))
            if config.Has("timeout") && IsInteger(config["timeout"])
                this.timeout := Max(100, Min(config["timeout"], 30000))
            if config.Has("enabled")
                this.enabled := !!config["enabled"]
        }

        ; Attempt to create the UIAutomation COM object
        try {
            this.uia := ComObject("UIAutomationClient.CUIAutomation")
            this._isAvailable := true
        } catch as err {
            this.uia := ""
            this._isAvailable := false
            ; Log warning but do not fail -- UIA is optional enhancement.
            ; On locked-down PCs or older Windows builds, COM creation can fail.
        }
    }

    ; =========================================================================
    ; Public Methods
    ; =========================================================================

    /**
     * Returns whether the UIA COM object was successfully created.
     * Callers should check this before requesting captures.
     *
     * @returns {Boolean} true if UIA is available and enabled
     */
    IsAvailable() {
        return this._isAvailable && this.enabled
    }

    /**
     * Captures UIA properties of the element at the given screen coordinates.
     * Called during recording at each click point to annotate click targets.
     *
     * @param {Integer} x - Screen X coordinate
     * @param {Integer} y - Screen Y coordinate
     * @returns {Map} Element properties, or Map with "error" key on failure
     */
    CaptureElementAtPoint(x, y) {
        if !this.IsAvailable()
            return Map("error", "UIA not available")

        try {
            ; ElementFromPoint expects a POINT structure (two 32-bit ints packed)
            pt := Buffer(8, 0)
            NumPut("Int", x, pt, 0)
            NumPut("Int", y, pt, 4)

            element := this.uia.ElementFromPoint(pt)
            if !element
                return Map("error", "No element found at point (" x ", " y ")")

            result := this.ElementToMap(element)
            return result
        } catch as err {
            return Map("error", "CaptureElementAtPoint failed: " err.Message)
        }
    }

    /**
     * Captures the UIA tree for a window, walking the accessibility hierarchy.
     * Returns a nested tree structure suitable for AI context enrichment.
     *
     * If hwnd is 0, captures the active foreground window.
     *
     * Performance note: At depth 5, this can take 100-2000ms depending on
     * application complexity. Call sparingly (window change, before submission).
     *
     * @param {Integer} hwnd     - Window handle (0 = active window)
     * @param {Integer} maxDepth - Override max depth (0 = use configured default)
     * @returns {Map} Tree structure with "window", "root", and "stats" keys
     */
    CaptureWindowTree(hwnd := 0, maxDepth := 0) {
        if !this.IsAvailable()
            return Map(
                "error", "UIA not available",
                "window", Map(),
                "root", Map(),
                "stats", Map("elementCount", 0, "depth", 0, "captureMs", 0)
            )

        captureStart := A_TickCount
        effectiveDepth := maxDepth > 0 ? maxDepth : this.maxDepth

        ; Reset per-capture counters
        this._elementCount := 0
        this._startTick := captureStart

        ; Resolve window handle
        windowTitle := ""
        windowClass := ""
        windowExe := ""
        windowPid := 0

        try {
            if hwnd = 0 {
                hwnd := WinGetID("A")
            }
            windowTitle := WinGetTitle(hwnd)
            windowClass := WinGetClass(hwnd)
            windowExe := WinGetProcessName(hwnd)
            windowPid := WinGetPID(hwnd)
        } catch as err {
            ; Window may have closed between capture request and now
            return Map(
                "error", "Failed to resolve window: " err.Message,
                "window", Map("hwnd", hwnd),
                "root", Map(),
                "stats", Map("elementCount", 0, "depth", 0, "captureMs", A_TickCount - captureStart)
            )
        }

        windowInfo := Map(
            "hwnd", hwnd,
            "title", windowTitle,
            "class", windowClass,
            "exe", windowExe,
            "pid", windowPid
        )

        ; Get root element for the window
        rootTree := Map()
        actualDepth := 0

        try {
            rootElement := this.uia.ElementFromHandle(hwnd)
            if !rootElement {
                return Map(
                    "error", "No UIA root element for window",
                    "window", windowInfo,
                    "root", Map(),
                    "stats", Map("elementCount", 0, "depth", 0, "captureMs", A_TickCount - captureStart)
                )
            }
            rootTree := this._WalkTree(rootElement, 0, effectiveDepth, &actualDepth)
        } catch as err {
            return Map(
                "error", "Tree walk failed: " err.Message,
                "window", windowInfo,
                "root", Map(),
                "stats", Map("elementCount", this._elementCount, "depth", 0, "captureMs", A_TickCount - captureStart)
            )
        }

        elapsedMs := A_TickCount - captureStart

        return Map(
            "window", windowInfo,
            "root", rootTree,
            "stats", Map(
                "elementCount", this._elementCount,
                "depth", actualDepth,
                "captureMs", elapsedMs
            )
        )
    }

    /**
     * Convenience method combining element-at-point with a shallow window tree
     * snippet. Called during recording on every mouse click to provide rich
     * context for the AI.
     *
     * @param {Integer} x    - Click X coordinate
     * @param {Integer} y    - Click Y coordinate
     * @param {Integer} hwnd - Window handle where click occurred
     * @returns {Map} Combined context with "clickedElement", "nearbyElements",
     *                "windowTitle", "windowClass"
     */
    CaptureClickContext(x, y, hwnd) {
        result := Map(
            "clickedElement", Map(),
            "nearbyElements", Array(),
            "windowTitle", "",
            "windowClass", ""
        )

        ; Get window info
        try {
            if hwnd != 0 {
                result["windowTitle"] := WinGetTitle(hwnd)
                result["windowClass"] := WinGetClass(hwnd)
            }
        } catch {
            ; Window may have closed
        }

        if !this.IsAvailable()
            return result

        ; Capture the clicked element
        result["clickedElement"] := this.CaptureElementAtPoint(x, y)

        ; Capture nearby/sibling elements for additional context
        try {
            pt := Buffer(8, 0)
            NumPut("Int", x, pt, 0)
            NumPut("Int", y, pt, 4)
            clickedEl := this.uia.ElementFromPoint(pt)

            if clickedEl {
                nearby := this._GetNearbyElements(clickedEl)
                result["nearbyElements"] := nearby
            }
        } catch {
            ; Non-critical: nearby elements are supplementary context
        }

        return result
    }

    /**
     * Extracts all useful properties from a UIA element into a plain Map.
     * Uses try/catch around each property access because some elements
     * throw on certain property reads (e.g., offscreen or destroyed elements).
     *
     * @param {ComObject} element - IUIAutomationElement COM object
     * @returns {Map} Element properties
     */
    ElementToMap(element) {
        result := Map()

        ; Each property access is individually guarded because UIA elements
        ; can throw on any property if the element is stale, offscreen,
        ; or from a protected process.

        try {
            result["name"] := element.CurrentName
        } catch {
            result["name"] := ""
        }

        try {
            result["className"] := element.CurrentClassName
        } catch {
            result["className"] := ""
        }

        try {
            result["automationId"] := element.CurrentAutomationId
        } catch {
            result["automationId"] := ""
        }

        try {
            typeId := element.CurrentControlType
            result["controlType"] := this.ControlTypeToString(typeId)
            result["controlTypeId"] := typeId
        } catch {
            result["controlType"] := "Unknown"
            result["controlTypeId"] := 0
        }

        ; Bounding rectangle -- returns a struct with left, top, right, bottom
        try {
            rect := element.CurrentBoundingRectangle
            ; rect is a RECT struct: the COM wrapper returns it as an object
            ; or we may need to read it as a structure depending on the COM interface
            result["boundingRect"] := this._ExtractRect(rect)
        } catch {
            result["boundingRect"] := Map("x", 0, "y", 0, "width", 0, "height", 0)
        }

        try {
            result["isEnabled"] := element.CurrentIsEnabled
        } catch {
            result["isEnabled"] := true
        }

        try {
            result["isOffscreen"] := element.CurrentIsOffscreen
        } catch {
            result["isOffscreen"] := false
        }

        try {
            result["helpText"] := element.CurrentHelpText
        } catch {
            result["helpText"] := ""
        }

        try {
            result["processId"] := element.CurrentProcessId
        } catch {
            result["processId"] := 0
        }

        ; Attempt to read Value via ValuePattern
        try {
            result["value"] := this._GetElementValue(element)
        } catch {
            result["value"] := ""
        }

        ; Attempt to read toggle state via TogglePattern
        try {
            result["toggleState"] := this._GetToggleState(element)
        } catch {
            result["toggleState"] := ""
        }

        ; Get supported interaction patterns
        try {
            result["patterns"] := this._GetSupportedPatterns(element)
        } catch {
            result["patterns"] := Array()
        }

        return result
    }

    /**
     * Converts a UIA control type ID to a human-readable string.
     *
     * @param {Integer} typeId - UIA control type constant (50000-50040)
     * @returns {String} Human-readable control type name
     */
    ControlTypeToString(typeId) {
        if UIACapture.ControlTypes.Has(typeId)
            return UIACapture.ControlTypes[typeId]
        return "Unknown(" typeId ")"
    }

    ; =========================================================================
    ; Internal Methods
    ; =========================================================================

    /**
     * Recursive depth-limited tree traversal using the RawViewWalker.
     * Respects maxDepth, maxElements, and timeout limits.
     *
     * @param {ComObject} element      - Current UIA element
     * @param {Integer}   currentDepth - Current recursion depth
     * @param {Integer}   maxDepth     - Maximum allowed depth
     * @param {VarRef}    actualDepth  - Tracks the deepest level reached
     * @returns {Map} Serialized element with "children" array
     */
    _WalkTree(element, currentDepth, maxDepth, &actualDepth := 0) {
        ; Check element count limit
        if this._elementCount >= this.maxElements
            return Map("_truncated", true, "reason", "maxElements reached")

        ; Check timeout
        if (A_TickCount - this._startTick) >= this.timeout
            return Map("_truncated", true, "reason", "timeout reached")

        ; Track deepest level
        if currentDepth > actualDepth
            actualDepth := currentDepth

        ; Serialize current element
        this._elementCount++
        node := this.ElementToMap(element)
        node["depth"] := currentDepth
        node["children"] := Array()

        ; Stop recursion at max depth
        if currentDepth >= maxDepth
            return node

        ; Walk children using TreeWalker
        try {
            walker := this.uia.RawViewWalker

            try {
                child := walker.GetFirstChildElement(element)
            } catch {
                child := ""
            }

            while child {
                ; Re-check limits before processing each child
                if this._elementCount >= this.maxElements
                    break
                if (A_TickCount - this._startTick) >= this.timeout
                    break

                try {
                    childNode := this._WalkTree(child, currentDepth + 1, maxDepth, &actualDepth)
                    node["children"].Push(childNode)
                } catch {
                    ; Individual child failure does not abort the walk
                }

                ; Get next sibling
                try {
                    child := walker.GetNextSiblingElement(child)
                } catch {
                    break
                }
            }
        } catch {
            ; Walker creation or traversal failed -- return node without children
        }

        return node
    }

    /**
     * Gets parent and sibling elements of a clicked element for context.
     * Returns a shallow list (not a deep tree) of nearby elements.
     *
     * @param {ComObject} element - The clicked UIA element
     * @returns {Array} Array of Maps representing nearby elements
     */
    _GetNearbyElements(element) {
        nearby := Array()

        try {
            walker := this.uia.RawViewWalker

            ; Get parent element
            try {
                parent := walker.GetParentElement(element)
                if parent {
                    parentMap := this.ElementToMap(parent)
                    parentMap["relation"] := "parent"
                    nearby.Push(parentMap)

                    ; Get siblings (other children of the parent)
                    siblingCount := 0
                    maxSiblings := 10  ; Limit sibling capture

                    try {
                        sibling := walker.GetFirstChildElement(parent)
                        while sibling && siblingCount < maxSiblings {
                            ; Skip the clicked element itself
                            try {
                                siblingRt := sibling.CurrentBoundingRectangle
                                elementRt := element.CurrentBoundingRectangle
                                ; Simple identity check via bounding rect
                                ; (UIA has no reliable object identity comparison via COM)
                                isSame := false
                                try {
                                    isSame := (this._ExtractRect(siblingRt)["x"] = this._ExtractRect(elementRt)["x"]
                                        && this._ExtractRect(siblingRt)["y"] = this._ExtractRect(elementRt)["y"])
                                } catch {
                                    isSame := false
                                }

                                if !isSame {
                                    sibMap := this.ElementToMap(sibling)
                                    sibMap["relation"] := "sibling"
                                    nearby.Push(sibMap)
                                    siblingCount++
                                }
                            } catch {
                                ; Skip problematic siblings
                            }

                            try {
                                sibling := walker.GetNextSiblingElement(sibling)
                            } catch {
                                break
                            }
                        }
                    } catch {
                        ; Failed to enumerate siblings
                    }
                }
            } catch {
                ; No parent available (root element)
            }
        } catch {
            ; Walker unavailable
        }

        return nearby
    }

    /**
     * Extracts a bounding rectangle from a UIA RECT structure.
     * The COM interface returns RECT as a tagged struct or array.
     * We handle multiple possible return formats defensively.
     *
     * @param {Object} rect - RECT from CurrentBoundingRectangle
     * @returns {Map} with keys: x, y, width, height
     */
    _ExtractRect(rect) {
        result := Map("x", 0, "y", 0, "width", 0, "height", 0)

        try {
            ; The COM UIA interface returns the bounding rect as a structure.
            ; In AHK v2 COM, this may come as a tagged union or a SAFEARRAY.
            ; We try multiple access patterns.

            ; Pattern 1: Direct struct property access (some COM wrappers)
            if IsObject(rect) {
                try {
                    result["x"] := rect.left
                    result["y"] := rect.top
                    result["width"] := rect.right - rect.left
                    result["height"] := rect.bottom - rect.top
                    return result
                } catch {
                }

                ; Pattern 2: Array-like indexed access
                try {
                    result["x"] := rect[0]
                    result["y"] := rect[1]
                    result["width"] := rect[2] - rect[0]
                    result["height"] := rect[3] - rect[1]
                    return result
                } catch {
                }
            }

            ; Pattern 3: VARIANT/SAFEARRAY returned as a ComValue
            ; The raw COM IUIAutomationElement::CurrentBoundingRectangle
            ; returns a RECT struct (4 doubles in UIA, or 4 ints).
            ; In AHK v2, this often appears as a tagged value.
            if !IsObject(rect) {
                ; Might be a pointer to a RECT buffer
                try {
                    result["x"] := NumGet(rect, 0, "Int")
                    result["y"] := NumGet(rect, 4, "Int")
                    result["width"] := NumGet(rect, 8, "Int") - NumGet(rect, 0, "Int")
                    result["height"] := NumGet(rect, 12, "Int") - NumGet(rect, 4, "Int")
                    return result
                } catch {
                }
            }
        } catch {
            ; Return zeroed rect
        }

        return result
    }

    /**
     * Attempts to read the Value property from an element via ValuePattern.
     * Not all elements support ValuePattern.
     *
     * UIA_ValuePatternId = 10002
     *
     * @param {ComObject} element - IUIAutomationElement
     * @returns {String} Element value or empty string
     */
    _GetElementValue(element) {
        try {
            ; IUIAutomationElement::GetCurrentPattern(UIA_ValuePatternId)
            ; UIA_ValuePatternId = 10002
            pattern := element.GetCurrentPattern(10002)
            if pattern {
                return pattern.CurrentValue
            }
        } catch {
        }
        return ""
    }

    /**
     * Attempts to read the ToggleState from an element via TogglePattern.
     * Returns "on", "off", "indeterminate", or empty string.
     *
     * UIA_TogglePatternId = 10015
     *
     * @param {ComObject} element - IUIAutomationElement
     * @returns {String} Toggle state string or empty
     */
    _GetToggleState(element) {
        try {
            ; UIA_TogglePatternId = 10015
            pattern := element.GetCurrentPattern(10015)
            if pattern {
                state := pattern.CurrentToggleState
                switch state {
                    case 0: return "off"
                    case 1: return "on"
                    case 2: return "indeterminate"
                    default: return "unknown(" state ")"
                }
            }
        } catch {
        }
        return ""
    }

    /**
     * Determines which UIA interaction patterns an element supports.
     * This tells the AI what operations are possible on the element
     * (click, set value, toggle, scroll, etc.).
     *
     * @param {ComObject} element - IUIAutomationElement
     * @returns {Array} Array of pattern name strings
     */
    _GetSupportedPatterns(element) {
        supported := Array()

        for patternId, patternName in UIACapture.PatternIds {
            try {
                pattern := element.GetCurrentPattern(patternId)
                if pattern {
                    supported.Push(patternName)
                }
            } catch {
                ; Pattern not supported or query failed -- skip
            }
        }

        return supported
    }
}
