;; ==========================================================================
;; Serializer.ahk -- Workflow Serializer & Action Merger for Claude AHK
;; ==========================================================================
;;
;; Transforms raw recorder events and UIA context into a structured,
;; protocol-compliant JSON payload for the Hercules proxy server.
;;
;; The core challenge: reduce thousands of raw input events (individual
;; keystrokes, polled mouse positions, window state changes) into a clean
;; sequence of high-level user actions that Claude can reason about.
;;
;; Requires: AutoHotkey v2.0.21+ (64-bit)
;; Depends:  vendor/Jxon.ahk (JSON serialization)
;; Module:   lib/Serializer.ahk
;; Version:  1.0.0
;;
;; Classes:
;;   WorkflowSerializer  -- Main serializer class (public API)
;;   SerializerError      -- Custom error type for this module
;;
;; Usage Example:
;;   config := Map(
;;       "keyGroupThreshold",   500,
;;       "mouseMergeThreshold", 200,
;;       "clickDedupeRadius",   3,
;;       "clickDedupeTimeMs",   200,
;;       "rdpEpsilon",          5.0,
;;       "maxActions",          1000,
;;       "maxDelayCapMs",       5000
;;   )
;;   serializer := WorkflowSerializer(config)
;;
;;   ; After recording stops, get events and UIA data
;;   events := recordingSession.GetEvents()
;;   uiaCaptures := Map()   ; timestamp -> CaptureClickContext result
;;
;;   ; Serialize into a protocol-compliant Map (ready for Jxon_Dump)
;;   workflow := serializer.SerializeWorkflow(events, uiaCaptures)
;;
;;   ; Convert to JSON string
;;   jsonStr := serializer.SerializeToJSON(workflow)
;;
;; Pipeline:
;;   1. Filter out meta events (session_start, session_stop, pause_marker)
;;   2. Merge consecutive keystrokes into "type" actions
;;   3. Detect key combos (modifier + key) into "key_combo" actions
;;   4. Merge consecutive mouse_move events and simplify paths (RDP)
;;   5. Pair mouse_down/mouse_up into "click" actions
;;   6. Detect double-clicks
;;   7. Merge scroll events at same position
;;   8. Convert window_change events into "window_switch" actions
;;   9. Attach UIA context to actions
;;  10. Deduplicate redundant actions
;;  11. Assign sequence numbers and compute timing deltas
;;  12. Build and validate the final payload
;;
;; ==========================================================================

#Requires AutoHotkey v2.0

#Include ..\vendor\Jxon.ahk

; ---------------------------------------------------------------------------
; SerializerError -- Custom error class for the Serializer module
; ---------------------------------------------------------------------------
class SerializerError extends Error {
    source := ""

    __New(message, source := "Serializer") {
        super.__New(message)
        this.source := source
    }
}

; ---------------------------------------------------------------------------
; WorkflowSerializer -- Main serializer class
; ---------------------------------------------------------------------------
; Converts raw Recorder events + UIACapture snapshots into the JSON payload
; defined in PROTOCOL.md section 3.1 (Workflow Submission).
; ---------------------------------------------------------------------------
class WorkflowSerializer {

    ; --- Configuration ---
    keyGroupThreshold   := 500      ; Max ms gap between keystrokes to merge into "type"
    mouseMergeThreshold := 200      ; Max ms gap between mouse_move events to merge
    clickDedupeRadius   := 3        ; Pixel radius for click deduplication
    clickDedupeTimeMs   := 200      ; Max ms between duplicate clicks
    rdpEpsilon          := 5.0      ; Ramer-Douglas-Peucker simplification epsilon (pixels)
    maxActions          := 1000     ; Protocol limit on actions per workflow
    maxDelayCapMs       := 5000     ; Maximum delay between actions to preserve
    doubleClickTimeMs   := 500      ; Max ms between two clicks for double-click detection
    doubleClickRadius   := 5        ; Pixel tolerance for double-click detection
    windowDedupeTimeMs  := 500      ; Max ms between identical window switches to dedupe

    ; --- Modifier key set (used for detecting key combos) ---
    static MODIFIER_KEYS := Map(
        "LCtrl",  "Ctrl",  "RCtrl",  "Ctrl",
        "LAlt",   "Alt",   "RAlt",   "Alt",
        "LShift", "Shift", "RShift", "Shift",
        "LWin",   "Win",   "RWin",   "Win",
        "Ctrl",   "Ctrl",  "Alt",    "Alt",
        "Shift",  "Shift", "Win",    "Win"
    )

    ; --- Constructor ---
    ; config : Map (optional) with configuration overrides
    __New(config := "") {
        if IsObject(config) {
            if config is Map {
                if config.Has("keyGroupThreshold")
                    this.keyGroupThreshold := this._ClampInt(config["keyGroupThreshold"], 50, 5000)
                if config.Has("mouseMergeThreshold")
                    this.mouseMergeThreshold := this._ClampInt(config["mouseMergeThreshold"], 50, 2000)
                if config.Has("clickDedupeRadius")
                    this.clickDedupeRadius := this._ClampInt(config["clickDedupeRadius"], 1, 50)
                if config.Has("clickDedupeTimeMs")
                    this.clickDedupeTimeMs := this._ClampInt(config["clickDedupeTimeMs"], 50, 2000)
                if config.Has("rdpEpsilon")
                    this.rdpEpsilon := Max(0.5, Min(config["rdpEpsilon"], 50.0))
                if config.Has("maxActions")
                    this.maxActions := this._ClampInt(config["maxActions"], 1, 1000)
                if config.Has("maxDelayCapMs")
                    this.maxDelayCapMs := this._ClampInt(config["maxDelayCapMs"], 500, 60000)
                if config.Has("doubleClickTimeMs")
                    this.doubleClickTimeMs := this._ClampInt(config["doubleClickTimeMs"], 100, 1000)
                if config.Has("doubleClickRadius")
                    this.doubleClickRadius := this._ClampInt(config["doubleClickRadius"], 1, 20)
                if config.Has("windowDedupeTimeMs")
                    this.windowDedupeTimeMs := this._ClampInt(config["windowDedupeTimeMs"], 100, 5000)
            }
        }
    }

    ; =====================================================================
    ; PUBLIC API
    ; =====================================================================

    ; SerializeWorkflow -- Main entry point
    ; Takes raw recorder events + UIA captures and produces a protocol-
    ; compliant Map suitable for Jxon_Dump and HTTP POST.
    ;
    ; events      : Array of Maps from RecordingSession.GetEvents()
    ;               Each Map has keys: "type", "time", "data"
    ; uiaCaptures : Map keyed by event time (Integer) -> CaptureClickContext result
    ;               (Optional -- empty Map if UIA was not enabled)
    ; Returns     : Map matching the protocol workflow schema
    SerializeWorkflow(events, uiaCaptures := "") {
        if !uiaCaptures
            uiaCaptures := Map()

        ; Validate inputs
        if !(events is Array)
            throw SerializerError("events must be an Array, got " Type(events))

        if events.Length = 0
            throw SerializerError("Cannot serialize empty event list")

        ; Extract session metadata from meta events before filtering
        sessionMeta := this._ExtractSessionMetadata(events)

        ; Step 1: Filter out meta events (session_start, session_stop, pause_marker)
        filtered := this._FilterMetaEvents(events)

        if filtered.Length = 0
            throw SerializerError("No actionable events after filtering meta events")

        ; Step 2: Build action sequence from raw events
        actions := this.BuildActionSequence(filtered, uiaCaptures)

        ; Step 3: Merge similar consecutive actions
        actions := this.MergeActions(actions)

        ; Step 4: Deduplicate redundant actions
        actions := this.DeduplicateActions(actions)

        ; Step 5: Enforce max actions limit
        if actions.Length > this.maxActions {
            trimmed := Array()
            loop this.maxActions {
                trimmed.Push(actions[A_Index])
            }
            actions := trimmed
        }

        ; Step 6: Assign sequence numbers and compute timing deltas
        actions := this._AssignSequenceAndTiming(actions)

        ; Step 7: Build the final payload
        workflow := this._BuildPayload(actions, sessionMeta)

        ; Step 8: Validate
        validation := this.ValidateWorkflow(workflow)
        if validation.Has("errors") && validation["errors"].Length > 0 {
            errMsg := "Workflow validation failed:"
            for _, e in validation["errors"] {
                errMsg .= " " e
            }
            throw SerializerError(errMsg)
        }

        return workflow
    }

    ; SerializeToJSON -- Converts a workflow Map to a JSON string
    ; workflow : Map from SerializeWorkflow()
    ; Returns  : String -- JSON text
    SerializeToJSON(workflow) {
        if !(workflow is Map)
            throw SerializerError("SerializeToJSON expects a Map, got " Type(workflow))

        try {
            return Jxon_Dump(workflow, 2)
        } catch as err {
            throw SerializerError("JSON serialization failed: " err.Message)
        }
    }

    ; BuildActionSequence -- Converts raw events into protocol action objects
    ; This is the core translation layer from recorder event types to
    ; PROTOCOL.md action types.
    ;
    ; events      : Array of filtered raw events (no meta events)
    ; uiaCaptures : Map of UIA context data keyed by time
    ; Returns     : Array of action Maps (not yet sequenced)
    BuildActionSequence(events, uiaCaptures) {
        actions := Array()

        ; Track window context throughout the event stream
        currentWindow := Map("title", "", "class", "", "exe", "", "pid", 0)
        previousWindow := Map("title", "", "class", "", "exe", "", "pid", 0)

        ; Track modifier key state for combo detection
        activeModifiers := Map()

        ; Keystroke accumulation buffer
        charBuffer := Array()
        charBufferStart := 0

        ; Mouse state tracking
        mouseDownState := Map()  ; button -> Map("time", t, "x", x, "y", y)

        ; Process events in chronological order
        idx := 1
        while idx <= events.Length {
            evt := events[idx]
            evtType := evt["type"]
            evtTime := evt["time"]
            evtData := evt["data"]

            ; --- Window change events ---
            if (evtType = "window_change") {
                ; Flush any pending keystroke buffer before window change
                if charBuffer.Length > 0 {
                    typeAction := this._FlushCharBuffer(charBuffer, charBufferStart, currentWindow)
                    if typeAction
                        actions.Push(typeAction)
                    charBuffer := Array()
                    charBufferStart := 0
                }

                previousWindow := this._CloneMap(currentWindow)
                currentWindow := Map(
                    "title", evtData.Has("title") ? evtData["title"] : "",
                    "class", evtData.Has("class") ? evtData["class"] : "",
                    "exe",   evtData.Has("exe")   ? evtData["exe"]   : "",
                    "pid",   evtData.Has("pid")   ? evtData["pid"]   : 0
                )

                ; Create window_switch action
                switchAction := Map(
                    "action_type", "window_switch",
                    "time", evtTime,
                    "window_context", this._CloneMap(currentWindow),
                    "target_element", Jxon_Null(),
                    "window_switch", Map(
                        "from_title", previousWindow["title"],
                        "from_exe",   previousWindow["exe"],
                        "to_title",   currentWindow["title"],
                        "to_exe",     currentWindow["exe"],
                        "method",     "alt_tab"
                    )
                )
                actions.Push(switchAction)
                idx++
                continue
            }

            ; --- Keyboard: modifier tracking ---
            if (evtType = "key_down") {
                keyName := evtData.Has("key") ? evtData["key"] : ""

                if WorkflowSerializer.MODIFIER_KEYS.Has(keyName) {
                    modName := WorkflowSerializer.MODIFIER_KEYS[keyName]
                    activeModifiers[modName] := evtTime
                    idx++
                    continue
                }

                ; Non-modifier key_down while modifiers are held = key_combo
                if activeModifiers.Count > 0 {
                    ; Flush any pending char buffer
                    if charBuffer.Length > 0 {
                        typeAction := this._FlushCharBuffer(charBuffer, charBufferStart, currentWindow)
                        if typeAction
                            actions.Push(typeAction)
                        charBuffer := Array()
                        charBufferStart := 0
                    }

                    mods := Array()
                    for modName, _ in activeModifiers {
                        mods.Push(modName)
                    }

                    ; Sort modifiers for deterministic output
                    mods := this._SortModifiers(mods)

                    uiaElement := this._FindUIAContext(evtTime, uiaCaptures, "focused")
                    comboAction := Map(
                        "action_type", "key_combo",
                        "time", evtTime,
                        "window_context", this._CloneMap(currentWindow),
                        "target_element", uiaElement,
                        "key_combo", Map(
                            "modifiers", mods,
                            "key", keyName
                        )
                    )
                    actions.Push(comboAction)
                    idx++
                    continue
                }

                ; Standalone key_down for non-printable keys (Enter, Escape, Tab, etc.)
                ; These will be handled as key_combo with empty modifiers if they
                ; appear outside of text typing context. But we let char events
                ; handle the typing merge; only standalone non-char keys go here.
                ; (We check later if a char event follows within a short window.)
                idx++
                continue
            }

            ; --- Keyboard: modifier release ---
            if (evtType = "key_up") {
                keyName := evtData.Has("key") ? evtData["key"] : ""
                if WorkflowSerializer.MODIFIER_KEYS.Has(keyName) {
                    modName := WorkflowSerializer.MODIFIER_KEYS[keyName]
                    if activeModifiers.Has(modName)
                        activeModifiers.Delete(modName)
                }
                idx++
                continue
            }

            ; --- Keyboard: character typed ---
            if (evtType = "char") {
                charVal := evtData.Has("char") ? evtData["char"] : ""

                ; Check time gap from last char in buffer
                if charBuffer.Length > 0 {
                    lastCharTime := charBuffer[charBuffer.Length]["time"]
                    gap := evtTime - lastCharTime

                    if gap > this.keyGroupThreshold {
                        ; Gap too large: flush the buffer and start a new group
                        typeAction := this._FlushCharBuffer(charBuffer, charBufferStart, currentWindow)
                        if typeAction
                            actions.Push(typeAction)
                        charBuffer := Array()
                        charBufferStart := 0
                    }
                }

                ; Start a new buffer if empty
                if charBuffer.Length = 0
                    charBufferStart := evtTime

                charBuffer.Push(Map("char", charVal, "time", evtTime))
                idx++
                continue
            }

            ; --- Mouse down ---
            if (evtType = "mouse_down") {
                ; Flush any pending keystroke buffer
                if charBuffer.Length > 0 {
                    typeAction := this._FlushCharBuffer(charBuffer, charBufferStart, currentWindow)
                    if typeAction
                        actions.Push(typeAction)
                    charBuffer := Array()
                    charBufferStart := 0
                }

                button := evtData.Has("button") ? evtData["button"] : "left"
                mx := evtData.Has("x") ? evtData["x"] : 0
                my := evtData.Has("y") ? evtData["y"] : 0

                mouseDownState[button] := Map("time", evtTime, "x", mx, "y", my)
                idx++
                continue
            }

            ; --- Mouse up ---
            if (evtType = "mouse_up") {
                button := evtData.Has("button") ? evtData["button"] : "left"
                mx := evtData.Has("x") ? evtData["x"] : 0
                my := evtData.Has("y") ? evtData["y"] : 0

                if mouseDownState.Has(button) {
                    downInfo := mouseDownState[button]

                    ; Build click action from paired mouse_down + mouse_up
                    uiaElement := this._FindUIAContext(downInfo["time"], uiaCaptures, "click")

                    clickAction := Map(
                        "action_type", "click",
                        "time", downInfo["time"],
                        "window_context", this._CloneMap(currentWindow),
                        "target_element", uiaElement,
                        "click", Map(
                            "button", button,
                            "click_count", 1,
                            "x", downInfo["x"],
                            "y", downInfo["y"]
                        )
                    )
                    actions.Push(clickAction)
                    mouseDownState.Delete(button)
                } else {
                    ; Orphan mouse_up without corresponding mouse_down
                    ; Still record as a click (best effort)
                    uiaElement := this._FindUIAContext(evtTime, uiaCaptures, "click")

                    clickAction := Map(
                        "action_type", "click",
                        "time", evtTime,
                        "window_context", this._CloneMap(currentWindow),
                        "target_element", uiaElement,
                        "click", Map(
                            "button", button,
                            "click_count", 1,
                            "x", mx,
                            "y", my
                        )
                    )
                    actions.Push(clickAction)
                }
                idx++
                continue
            }

            ; --- Mouse move ---
            if (evtType = "mouse_move") {
                ; Collect consecutive mouse_move events
                movePoints := Array()
                moveStartTime := evtTime
                moveEndTime := evtTime

                while idx <= events.Length && events[idx]["type"] = "mouse_move" {
                    mEvt := events[idx]
                    mData := mEvt["data"]
                    px := mData.Has("x") ? mData["x"] : 0
                    py := mData.Has("y") ? mData["y"] : 0
                    movePoints.Push(Array(px, py))
                    moveEndTime := mEvt["time"]
                    idx++

                    ; Check time gap to next event
                    if idx <= events.Length && events[idx]["type"] = "mouse_move" {
                        gap := events[idx]["time"] - moveEndTime
                        if gap > this.mouseMergeThreshold
                            break
                    }
                }

                ; Only record if we have at least 2 points
                if movePoints.Length >= 2 {
                    ; Simplify the path using Ramer-Douglas-Peucker
                    simplified := this._SimplifyPath(movePoints, this.rdpEpsilon)

                    startPt := simplified[1]
                    endPt := simplified[simplified.Length]
                    durationMs := moveEndTime - moveStartTime

                    moveAction := Map(
                        "action_type", "mouse_move",
                        "time", moveStartTime,
                        "window_context", this._CloneMap(currentWindow),
                        "target_element", Jxon_Null(),
                        "mouse_move", Map(
                            "from_x", startPt[1],
                            "from_y", startPt[2],
                            "to_x", endPt[1],
                            "to_y", endPt[2],
                            "duration_ms", durationMs > 0 ? durationMs : 1
                        )
                    )
                    actions.Push(moveAction)
                }
                continue  ; idx already advanced by inner while loop
            }

            ; --- Mouse scroll ---
            if (evtType = "mouse_scroll") {
                direction := evtData.Has("direction") ? evtData["direction"] : "down"
                sx := evtData.Has("x") ? evtData["x"] : 0
                sy := evtData.Has("y") ? evtData["y"] : 0

                ; Accumulate consecutive scroll events in the same direction at same position
                scrollDelta := direction = "up" ? 1 : -1
                scrollTime := evtTime
                nextIdx := idx + 1

                while nextIdx <= events.Length && events[nextIdx]["type"] = "mouse_scroll" {
                    nEvt := events[nextIdx]
                    nData := nEvt["data"]
                    nDir := nData.Has("direction") ? nData["direction"] : "down"
                    nx := nData.Has("x") ? nData["x"] : 0
                    ny := nData.Has("y") ? nData["y"] : 0

                    ; Same direction and approximate position?
                    if (nDir = direction) && (Abs(nx - sx) <= 10) && (Abs(ny - sy) <= 10) {
                        scrollDelta += (nDir = "up" ? 1 : -1)
                        nextIdx++
                    } else {
                        break
                    }
                }
                idx := nextIdx

                uiaElement := this._FindUIAContext(scrollTime, uiaCaptures, "scroll")
                scrollAction := Map(
                    "action_type", "scroll",
                    "time", scrollTime,
                    "window_context", this._CloneMap(currentWindow),
                    "target_element", uiaElement,
                    "scroll", Map(
                        "x", sx,
                        "y", sy,
                        "delta", scrollDelta,
                        "direction", "vertical"
                    )
                )
                actions.Push(scrollAction)
                continue  ; idx already advanced
            }

            ; --- Unknown event type: skip ---
            idx++
        }

        ; Flush any remaining char buffer
        if charBuffer.Length > 0 {
            typeAction := this._FlushCharBuffer(charBuffer, charBufferStart, currentWindow)
            if typeAction
                actions.Push(typeAction)
        }

        return actions
    }

    ; MergeActions -- Merge consecutive similar actions
    ; Handles:
    ;   - Consecutive clicks at same position -> double-click
    ;   - (Keystroke merging is already handled in BuildActionSequence)
    ;   - (Mouse move merging is already handled in BuildActionSequence)
    ;
    ; actions : Array of action Maps
    ; Returns : Array of merged action Maps
    MergeActions(actions) {
        if actions.Length <= 1
            return actions

        merged := Array()
        idx := 1

        while idx <= actions.Length {
            current := actions[idx]

            ; --- Double-click detection ---
            if current["action_type"] = "click" && idx + 1 <= actions.Length {
                next := actions[idx + 1]

                if next["action_type"] = "click" {
                    cClick := current["click"]
                    nClick := next["click"]

                    timeDiff := next["time"] - current["time"]
                    dx := Abs(cClick["x"] - nClick["x"])
                    dy := Abs(cClick["y"] - nClick["y"])

                    ; Same button, close in time and space -> double-click
                    if (cClick["button"] = nClick["button"])
                        && (timeDiff <= this.doubleClickTimeMs)
                        && (dx <= this.doubleClickRadius)
                        && (dy <= this.doubleClickRadius) {

                        ; Replace both clicks with a single double-click
                        current["click"]["click_count"] := 2
                        merged.Push(current)
                        idx += 2
                        continue
                    }
                }
            }

            merged.Push(current)
            idx++
        }

        return merged
    }

    ; DeduplicateActions -- Remove redundant consecutive actions
    ; Handles:
    ;   - Multiple clicks on same element within short time
    ;   - Consecutive switches to same window
    ;
    ; actions : Array of action Maps
    ; Returns : Array of deduplicated action Maps
    DeduplicateActions(actions) {
        if actions.Length <= 1
            return actions

        deduped := Array()
        deduped.Push(actions[1])

        idx := 2
        while idx <= actions.Length {
            current := actions[idx]
            previous := deduped[deduped.Length]

            isDupe := false

            ; --- Click deduplication ---
            if (current["action_type"] = "click" && previous["action_type"] = "click") {
                cClick := current["click"]
                pClick := previous["click"]

                timeDiff := current["time"] - previous["time"]
                dx := Abs(cClick["x"] - pClick["x"])
                dy := Abs(cClick["y"] - pClick["y"])

                ; Same button, same approximate position, within dedup window
                ; but NOT a double-click (those already have click_count=2)
                if (cClick["button"] = pClick["button"])
                    && (cClick["click_count"] = 1 && pClick["click_count"] = 1)
                    && (timeDiff <= this.clickDedupeTimeMs)
                    && (dx <= this.clickDedupeRadius)
                    && (dy <= this.clickDedupeRadius) {
                    isDupe := true
                }
            }

            ; --- Window switch deduplication ---
            if (current["action_type"] = "window_switch" && previous["action_type"] = "window_switch") {
                cSwitch := current["window_switch"]
                pSwitch := previous["window_switch"]

                timeDiff := current["time"] - previous["time"]

                if (cSwitch["to_title"] = pSwitch["to_title"])
                    && (cSwitch["to_exe"] = pSwitch["to_exe"])
                    && (timeDiff <= this.windowDedupeTimeMs) {
                    isDupe := true
                }
            }

            if !isDupe
                deduped.Push(current)

            idx++
        }

        return deduped
    }

    ; ValidateWorkflow -- Validates a workflow Map against protocol constraints
    ; Returns : Map("valid", 1|0, "errors", Array(), "warnings", Array())
    ValidateWorkflow(workflow) {
        errors := Array()
        warnings := Array()

        ; Check top-level required keys
        if !(workflow is Map) {
            errors.Push("Workflow must be a Map")
            return Map("valid", 0, "errors", errors, "warnings", warnings)
        }

        if !workflow.Has("actions")
            errors.Push("Missing required key: actions")

        if !workflow.Has("metadata")
            errors.Push("Missing required key: metadata")

        if workflow.Has("actions") {
            actionsArr := workflow["actions"]

            if !(actionsArr is Array) {
                errors.Push("actions must be an Array")
            } else {
                ; Check action count limit
                if actionsArr.Length > this.maxActions
                    errors.Push("Action count " actionsArr.Length " exceeds limit of " this.maxActions)

                if actionsArr.Length = 0
                    warnings.Push("Workflow has zero actions")

                ; Validate each action
                validTypes := Map(
                    "click", 1, "type", 1, "key_combo", 1,
                    "mouse_move", 1, "scroll", 1, "window_switch", 1
                )

                for _, action in actionsArr {
                    if !(action is Map) {
                        errors.Push("Action at seq " _ " is not a Map")
                        continue
                    }

                    if !action.Has("action_type") {
                        errors.Push("Action missing action_type at index " _)
                        continue
                    }

                    aType := action["action_type"]
                    if !validTypes.Has(aType)
                        errors.Push("Invalid action_type '" aType "' at seq " (action.Has("seq") ? action["seq"] : _))

                    if !action.Has("seq")
                        errors.Push("Action missing seq at index " _)

                    if !action.Has("window_context")
                        warnings.Push("Action missing window_context at seq " (action.Has("seq") ? action["seq"] : _))

                    ; Validate type-specific payload exists
                    if (aType = "click" && !action.Has("click"))
                        errors.Push("click action missing 'click' payload at seq " (action.Has("seq") ? action["seq"] : _))
                    if (aType = "type" && !action.Has("type_text"))
                        errors.Push("type action missing 'type_text' payload at seq " (action.Has("seq") ? action["seq"] : _))
                    if (aType = "key_combo" && !action.Has("key_combo"))
                        errors.Push("key_combo action missing 'key_combo' payload at seq " (action.Has("seq") ? action["seq"] : _))
                    if (aType = "mouse_move" && !action.Has("mouse_move"))
                        errors.Push("mouse_move action missing 'mouse_move' payload at seq " (action.Has("seq") ? action["seq"] : _))
                    if (aType = "scroll" && !action.Has("scroll"))
                        errors.Push("scroll action missing 'scroll' payload at seq " (action.Has("seq") ? action["seq"] : _))
                    if (aType = "window_switch" && !action.Has("window_switch"))
                        errors.Push("window_switch action missing 'window_switch' payload at seq " (action.Has("seq") ? action["seq"] : _))
                }
            }
        }

        ; Check metadata
        if workflow.Has("metadata") {
            meta := workflow["metadata"]
            if !(meta is Map)
                errors.Push("metadata must be a Map")
            else {
                if !meta.Has("recorded_at")
                    warnings.Push("metadata missing recorded_at")
                if !meta.Has("duration_ms")
                    warnings.Push("metadata missing duration_ms")
                if !meta.Has("action_count")
                    warnings.Push("metadata missing action_count")
            }
        }

        isValid := errors.Length = 0 ? 1 : 0
        return Map("valid", isValid, "errors", errors, "warnings", warnings)
    }

    ; =====================================================================
    ; INTERNAL METHODS
    ; =====================================================================

    ; _FilterMetaEvents -- Removes session_start, session_stop, and pause_marker events
    ; These are recording bookkeeping, not user actions.
    _FilterMetaEvents(events) {
        metaTypes := Map("session_start", 1, "session_stop", 1, "pause_marker", 1)
        filtered := Array()

        for _, evt in events {
            if !metaTypes.Has(evt["type"])
                filtered.Push(evt)
        }

        return filtered
    }

    ; _ExtractSessionMetadata -- Pulls timing info from session_start/session_stop events
    _ExtractSessionMetadata(events) {
        meta := Map(
            "duration_ms", 0,
            "total_events", events.Length,
            "pause_total_ms", 0
        )

        for _, evt in events {
            if evt["type"] = "session_stop" && evt.Has("data") {
                data := evt["data"]
                if data.Has("duration")
                    meta["duration_ms"] := data["duration"]
                if data.Has("pause_total")
                    meta["pause_total_ms"] := data["pause_total"]
                if data.Has("event_count")
                    meta["total_events"] := data["event_count"]
            }
        }

        ; If no session_stop marker, estimate from last event time
        if meta["duration_ms"] = 0 && events.Length > 0 {
            meta["duration_ms"] := events[events.Length]["time"]
        }

        return meta
    }

    ; _FlushCharBuffer -- Converts accumulated char events into a "type" action
    ; charBuffer : Array of Map("char", c, "time", t)
    ; startTime  : time of first char
    ; windowCtx  : current window context Map
    ; Returns    : Map (type action) or "" if buffer is empty
    _FlushCharBuffer(charBuffer, startTime, windowCtx) {
        if charBuffer.Length = 0
            return ""

        text := ""
        for _, entry in charBuffer {
            text .= entry["char"]
        }

        ; Calculate typing speed (characters per second)
        duration := charBuffer[charBuffer.Length]["time"] - startTime
        speedCps := 0.0
        if duration > 0 && StrLen(text) > 1
            speedCps := Round((StrLen(text) / (duration / 1000.0)), 1)

        typeAction := Map(
            "action_type", "type",
            "time", startTime,
            "window_context", this._CloneMap(windowCtx),
            "target_element", Jxon_Null(),
            "type_text", Map(
                "text", text,
                "speed_cps", speedCps
            )
        )

        return typeAction
    }

    ; _AssignSequenceAndTiming -- Adds seq, timestamp, timing_delta_ms to each action
    ; actions : Array of action Maps (must have "time" key)
    ; Returns : Array of enriched action Maps matching protocol format
    _AssignSequenceAndTiming(actions) {
        if actions.Length = 0
            return actions

        previousTime := 0
        recordingStartTime := actions.Length > 0 ? actions[1]["time"] : 0

        for idx, action in actions {
            ; Zero-based sequence number
            action["seq"] := idx - 1

            ; ISO 8601 timestamp (relative to "now" as a placeholder;
            ; in real use, the recording start wall-clock time would be captured)
            action["timestamp"] := this._FormatISO8601()

            ; Timing delta from previous action
            if idx = 1 {
                action["timing_delta_ms"] := 0
            } else {
                rawDelta := action["time"] - previousTime
                ; Cap maximum delay
                action["timing_delta_ms"] := Min(rawDelta, this.maxDelayCapMs)
            }

            previousTime := action["time"]
        }

        return actions
    }

    ; _BuildPayload -- Constructs the final protocol-compliant payload Map
    ; actions     : Array of fully enriched action Maps
    ; sessionMeta : Map with session-level metadata
    ; Returns     : Map matching PROTOCOL.md workflow schema
    _BuildPayload(actions, sessionMeta) {
        ; Build the actions array with only protocol-required fields
        cleanActions := Array()
        for _, action in actions {
            clean := Map(
                "seq", action["seq"],
                "timestamp", action["timestamp"],
                "timing_delta_ms", action["timing_delta_ms"],
                "action_type", action["action_type"],
                "window_context", action.Has("window_context") ? action["window_context"] : Map("title", "", "class", "", "exe", "", "pid", 0),
                "target_element", action.Has("target_element") ? action["target_element"] : Jxon_Null()
            )

            ; Add type-specific payload
            aType := action["action_type"]
            if (aType = "click" && action.Has("click"))
                clean["click"] := action["click"]
            if (aType = "type" && action.Has("type_text"))
                clean["type_text"] := action["type_text"]
            if (aType = "key_combo" && action.Has("key_combo"))
                clean["key_combo"] := action["key_combo"]
            if (aType = "mouse_move" && action.Has("mouse_move"))
                clean["mouse_move"] := action["mouse_move"]
            if (aType = "scroll" && action.Has("scroll"))
                clean["scroll"] := action["scroll"]
            if (aType = "window_switch" && action.Has("window_switch"))
                clean["window_switch"] := action["window_switch"]

            cleanActions.Push(clean)
        }

        ; Determine the primary window context from the most common window
        primaryWindow := this._DeterminePrimaryWindow(actions)

        ; Compute recorded_at as ISO 8601
        recordedAt := this._FormatISO8601()

        ; Determine app context from action windows
        appContext := this._BuildAppContext(actions)

        ; Assemble the top-level workflow Map
        workflow := Map(
            "actions", cleanActions,
            "metadata", Map(
                "recorded_at", recordedAt,
                "duration_ms", sessionMeta["duration_ms"],
                "action_count", cleanActions.Length,
                "app_context", appContext
            ),
            "window_context", primaryWindow
        )

        return workflow
    }

    ; _FindUIAContext -- Finds the best matching UIA context for a given event time
    ; eventTime   : Integer -- relative time of the event
    ; uiaCaptures : Map keyed by time -> UIA snapshot
    ; contextType : String -- "click", "focused", "scroll" (hint for lookup)
    ; Returns     : Map (protocol target_element) or Jxon_Null()
    _FindUIAContext(eventTime, uiaCaptures, contextType := "click") {
        if !(uiaCaptures is Map) || uiaCaptures.Count = 0
            return Jxon_Null()

        ; Find the closest UIA snapshot by timestamp (within 500ms tolerance)
        bestMatch := ""
        bestDelta := 999999

        for captureTime, captureData in uiaCaptures {
            delta := Abs(eventTime - captureTime)
            if delta < bestDelta {
                bestDelta := delta
                bestMatch := captureData
            }
        }

        ; Only use if within 500ms
        if bestDelta > 500 || bestMatch = ""
            return Jxon_Null()

        ; Extract the element from CaptureClickContext format
        element := ""
        if bestMatch is Map {
            if bestMatch.Has("clickedElement")
                element := bestMatch["clickedElement"]
            else
                element := bestMatch
        }

        if !(element is Map) || element.Count = 0
            return Jxon_Null()

        ; Check for error in UIA capture
        if element.Has("error")
            return Jxon_Null()

        ; Map UIACapture format to protocol target_element format
        return this._MapUIAToProtocol(element)
    }

    ; _MapUIAToProtocol -- Converts UIACapture element Map to protocol target_element format
    ; uiaElement : Map from UIACapture.ElementToMap()
    ; Returns    : Map matching protocol target_element schema
    _MapUIAToProtocol(uiaElement) {
        if !(uiaElement is Map) || uiaElement.Count = 0
            return Jxon_Null()

        name := uiaElement.Has("name") ? uiaElement["name"] : ""

        ; Map controlType to protocol "role" field
        role := ""
        if uiaElement.Has("controlType")
            role := uiaElement["controlType"]
        else if uiaElement.Has("type")
            role := uiaElement["type"]

        automationId := uiaElement.Has("automationId") ? uiaElement["automationId"] : ""
        className := uiaElement.Has("className") ? uiaElement["className"] : ""

        ; Extract bounding rect
        boundingRect := Map("x", 0, "y", 0, "w", 0, "h", 0)
        if uiaElement.Has("boundingRect") && uiaElement["boundingRect"] is Map {
            br := uiaElement["boundingRect"]
            boundingRect["x"] := br.Has("x") ? br["x"] : 0
            boundingRect["y"] := br.Has("y") ? br["y"] : 0
            ; UIACapture uses "width"/"height" but protocol uses "w"/"h"
            if br.Has("w")
                boundingRect["w"] := br["w"]
            else if br.Has("width")
                boundingRect["w"] := br["width"]
            if br.Has("h")
                boundingRect["h"] := br["h"]
            else if br.Has("height")
                boundingRect["h"] := br["height"]
        }

        ; Value
        value := Jxon_Null()
        if uiaElement.Has("value") && uiaElement["value"] != ""
            value := uiaElement["value"]

        return Map(
            "name", name,
            "role", role,
            "automation_id", automationId,
            "class_name", className,
            "bounding_rect", boundingRect,
            "value", value
        )
    }

    ; _DeterminePrimaryWindow -- Finds the most frequently referenced window
    ; actions : Array of action Maps
    ; Returns : Map with title, class, process keys
    _DeterminePrimaryWindow(actions) {
        if actions.Length = 0
            return Map("title", "", "class", "", "process", "")

        ; Count window references by exe
        windowCounts := Map()
        lastWindow := Map("title", "", "class", "", "exe", "")

        for _, action in actions {
            if action.Has("window_context") && action["window_context"] is Map {
                wc := action["window_context"]
                exe := wc.Has("exe") ? wc["exe"] : ""
                if exe != "" {
                    if windowCounts.Has(exe)
                        windowCounts[exe] := windowCounts[exe] + 1
                    else
                        windowCounts[exe] := 1
                    lastWindow := wc
                }
            }
        }

        ; Find the most common exe
        maxCount := 0
        primaryExe := ""
        for exe, count in windowCounts {
            if count > maxCount {
                maxCount := count
                primaryExe := exe
            }
        }

        ; Find the most recent window context for that exe
        primaryCtx := Map("title", "", "class", "", "process", "")
        loop actions.Length {
            reverseIdx := actions.Length - A_Index + 1
            action := actions[reverseIdx]
            if action.Has("window_context") && action["window_context"] is Map {
                wc := action["window_context"]
                if (wc.Has("exe") && wc["exe"] = primaryExe) {
                    primaryCtx["title"]   := wc.Has("title") ? wc["title"] : ""
                    primaryCtx["class"]   := wc.Has("class") ? wc["class"] : ""
                    primaryCtx["process"] := primaryExe
                    break
                }
            }
        }

        return primaryCtx
    }

    ; _BuildAppContext -- Creates app_context metadata from action windows
    ; actions : Array of action Maps
    ; Returns : Map with unique applications encountered
    _BuildAppContext(actions) {
        apps := Map()

        for _, action in actions {
            if action.Has("window_context") && action["window_context"] is Map {
                wc := action["window_context"]
                exe := wc.Has("exe") ? wc["exe"] : ""
                if exe != "" && !apps.Has(exe) {
                    apps[exe] := Map(
                        "exe", exe,
                        "title", wc.Has("title") ? wc["title"] : "",
                        "class", wc.Has("class") ? wc["class"] : ""
                    )
                }
            }
        }

        ; Convert to Array
        appList := Array()
        for _, app in apps {
            appList.Push(app)
        }

        return Map(
            "applications", appList,
            "app_count", appList.Length
        )
    }

    ; =====================================================================
    ; PATH SIMPLIFICATION (Ramer-Douglas-Peucker)
    ; =====================================================================

    ; _SimplifyPath -- Reduces point count while preserving shape
    ; points  : Array of [x, y] pairs (each element is an Array(x, y))
    ; epsilon : Distance threshold in pixels
    ; Returns : Simplified Array of [x, y] pairs
    _SimplifyPath(points, epsilon := 5.0) {
        if points.Length <= 2
            return points

        ; Find the point with the maximum distance from the line between
        ; the first and last points
        maxDist := 0.0
        maxIdx := 0
        startPt := points[1]
        endPt := points[points.Length]

        loop points.Length - 2 {
            idx := A_Index + 1
            dist := this._PerpendicularDistance(points[idx], startPt, endPt)
            if dist > maxDist {
                maxDist := dist
                maxIdx := idx
            }
        }

        ; If max distance is greater than epsilon, recursively simplify
        if maxDist > epsilon {
            ; Split into two segments and simplify each
            leftSegment := Array()
            loop maxIdx {
                leftSegment.Push(points[A_Index])
            }

            rightSegment := Array()
            loop points.Length - maxIdx + 1 {
                rightSegment.Push(points[maxIdx + A_Index - 1])
            }

            leftSimplified := this._SimplifyPath(leftSegment, epsilon)
            rightSimplified := this._SimplifyPath(rightSegment, epsilon)

            ; Merge results (remove duplicate point at junction)
            result := Array()
            loop leftSimplified.Length - 1 {
                result.Push(leftSimplified[A_Index])
            }
            for _, pt in rightSimplified {
                result.Push(pt)
            }

            return result
        } else {
            ; All intermediate points are close enough; keep only endpoints
            return Array(startPt, endPt)
        }
    }

    ; _PerpendicularDistance -- Distance from a point to the line through two endpoints
    ; point    : Array(x, y)
    ; lineStart: Array(x, y)
    ; lineEnd  : Array(x, y)
    ; Returns  : Float -- perpendicular distance in pixels
    _PerpendicularDistance(point, lineStart, lineEnd) {
        px := point[1]
        py := point[2]
        x1 := lineStart[1]
        y1 := lineStart[2]
        x2 := lineEnd[1]
        y2 := lineEnd[2]

        ; Length of the line segment
        dx := x2 - x1
        dy := y2 - y1
        lineLenSq := dx * dx + dy * dy

        ; If line has zero length, return distance to the start point
        if lineLenSq = 0
            return Sqrt((px - x1) * (px - x1) + (py - y1) * (py - y1))

        ; Perpendicular distance using cross product formula:
        ; |distance| = |((y2-y1)*px - (x2-x1)*py + x2*y1 - y2*x1)| / sqrt((y2-y1)^2 + (x2-x1)^2)
        numerator := Abs(dy * px - dx * py + x2 * y1 - y2 * x1)
        denominator := Sqrt(lineLenSq)

        return numerator / denominator
    }

    ; =====================================================================
    ; UTILITY METHODS
    ; =====================================================================

    ; _CloneMap -- Shallow clone a Map
    ; m       : Map to clone
    ; Returns : new Map with same key-value pairs
    _CloneMap(m) {
        if !(m is Map)
            return Map()

        clone := Map()
        for k, v in m {
            clone[k] := v
        }
        return clone
    }

    ; _ClampInt -- Clamp an integer value between min and max
    ; val     : value to clamp
    ; minVal  : minimum
    ; maxVal  : maximum
    ; Returns : Integer
    _ClampInt(val, minVal, maxVal) {
        v := Integer(val)
        if v < minVal
            return minVal
        if v > maxVal
            return maxVal
        return v
    }

    ; _SortModifiers -- Sort modifier names into a canonical order
    ; mods    : Array of modifier strings
    ; Returns : Array sorted as [Ctrl, Alt, Shift, Win]
    _SortModifiers(mods) {
        order := Map("Ctrl", 1, "Alt", 2, "Shift", 3, "Win", 4)
        sorted := Array()

        ; Deduplicate first
        seen := Map()
        for _, m in mods {
            if !seen.Has(m) {
                seen[m] := 1
            }
        }

        ; Insert in canonical order
        loop 4 {
            switch A_Index {
                case 1:
                    if seen.Has("Ctrl")
                        sorted.Push("Ctrl")
                case 2:
                    if seen.Has("Alt")
                        sorted.Push("Alt")
                case 3:
                    if seen.Has("Shift")
                        sorted.Push("Shift")
                case 4:
                    if seen.Has("Win")
                        sorted.Push("Win")
            }
        }

        return sorted
    }

    ; _FormatISO8601 -- Produces an ISO 8601 UTC timestamp string
    ; Uses the current system time. In production, the recording start
    ; wall-clock time would be captured and offsets computed.
    ; Returns : String in format "2026-02-13T14:30:00.000Z"
    _FormatISO8601() {
        ; AHK v2 FormatTime uses the YYYYMMDD... format codes
        ; We build ISO 8601 manually for UTC output
        now := A_NowUTC
        year  := SubStr(now, 1, 4)
        month := SubStr(now, 5, 2)
        day   := SubStr(now, 7, 2)
        hour  := SubStr(now, 9, 2)
        min   := SubStr(now, 11, 2)
        sec   := SubStr(now, 13, 2)

        return year "-" month "-" day "T" hour ":" min ":" sec ".000Z"
    }

    ; GetStatistics -- Returns stats about the last serialization
    ; (Useful for debugging and GUI display)
    ; workflow : Map from SerializeWorkflow()
    ; Returns  : Map with readable statistics
    static GetStatistics(workflow) {
        stats := Map(
            "action_count", 0,
            "duration_ms", 0,
            "action_types", Map(),
            "window_count", 0,
            "has_uia_context", 0
        )

        if !(workflow is Map)
            return stats

        if workflow.Has("metadata") && workflow["metadata"] is Map {
            meta := workflow["metadata"]
            if meta.Has("action_count")
                stats["action_count"] := meta["action_count"]
            if meta.Has("duration_ms")
                stats["duration_ms"] := meta["duration_ms"]
        }

        if workflow.Has("actions") && workflow["actions"] is Array {
            typeCounts := Map()
            uiaCount := 0
            windows := Map()

            for _, action in workflow["actions"] {
                if action.Has("action_type") {
                    aType := action["action_type"]
                    if typeCounts.Has(aType)
                        typeCounts[aType] := typeCounts[aType] + 1
                    else
                        typeCounts[aType] := 1
                }

                if action.Has("target_element") {
                    te := action["target_element"]
                    if te is Map && te.Count > 0
                        uiaCount++
                }

                if action.Has("window_context") && action["window_context"] is Map {
                    wc := action["window_context"]
                    if wc.Has("title") && wc["title"] != ""
                        windows[wc["title"]] := 1
                }
            }

            stats["action_types"] := typeCounts
            stats["has_uia_context"] := uiaCount
            stats["window_count"] := windows.Count
        }

        return stats
    }

    ; EstimatePayloadSize -- Estimates the JSON payload size in bytes
    ; workflow : Map from SerializeWorkflow()
    ; Returns  : Integer -- approximate byte count
    static EstimatePayloadSize(workflow) {
        try {
            json := Jxon_Dump(workflow)
            ; Approximate UTF-8 byte size (ASCII-dominated content)
            return StrLen(json)
        } catch {
            return 0
        }
    }
}
