;; ==========================================================================
;; Recorder.ahk -- Workflow Recording Module for Claude AHK
;; ==========================================================================
;;
;; Captures raw user input events: keystrokes, mouse actions, and active
;; window changes. Composed of three specialized sub-recorders managed by
;; a session orchestrator (RecordingSession).
;;
;; Requires: AutoHotkey v2.0.21+ (64-bit)
;; Module:   lib/Recorder.ahk
;; Version:  1.0.0
;;
;; Classes:
;;   RecordingSession  -- Orchestrates all recorders, manages event buffer
;;   KeyboardRecorder  -- InputHook-based keystroke capture
;;   MouseRecorder     -- Hotkey + polling-based mouse capture
;;   WindowTracker     -- Polling-based active window tracking
;;   RecorderError     -- Custom error type for this module
;;
;; Usage Example:
;;   ; Create a recording session with default settings
;;   session := RecordingSession()
;;
;;   ; Start recording (keyboard, mouse, and window tracking all begin)
;;   session.Start()
;;
;;   ; ... user performs actions ...
;;
;;   ; Pause recording temporarily
;;   session.Pause()
;;
;;   ; Resume recording
;;   session.Resume()
;;
;;   ; Stop recording and retrieve all captured events
;;   events := session.Stop()
;;
;;   ; events is an Array of Maps, each with keys: "type", "time", "data"
;;   ; Example event:
;;   ;   Map("type", "key_down",
;;   ;       "time", 1523,
;;   ;       "data", Map("vk", 65, "sc", 30, "key", "a"))
;;
;;   ; Inspect event count and duration
;;   MsgBox "Captured " events.Length " events"
;;
;; Emergency Stop:
;;   The RecordingSession registers Ctrl+Shift+X as an emergency stop
;;   hotkey that immediately halts all recording.
;;
;; Event Types:
;;   key_down       -- Key pressed     (vk, sc, key)
;;   key_up         -- Key released    (vk, sc, key)
;;   char           -- Character typed (char)
;;   mouse_down     -- Mouse button pressed  (button, x, y, hwnd)
;;   mouse_up       -- Mouse button released (button, x, y, hwnd)
;;   mouse_move     -- Mouse cursor moved    (x, y)
;;   mouse_scroll   -- Mouse wheel scrolled  (direction, x, y)
;;   window_change  -- Active window changed (hwnd, title, class, exe, pid)
;;   pause_marker   -- Recording paused      (pause_duration)
;;   session_start  -- Recording began       ()
;;   session_stop   -- Recording ended       (event_count, duration)
;;
;; ==========================================================================

#Requires AutoHotkey v2.0

; ---------------------------------------------------------------------------
; RecorderError -- Custom error class for the Recorder module
; ---------------------------------------------------------------------------
class RecorderError extends Error {
    source := ""

    __New(message, source := "Recorder") {
        super.__New(message)
        this.source := source
    }
}

; ---------------------------------------------------------------------------
; KeyboardRecorder -- Captures keystrokes via AHK v2 InputHook
; ---------------------------------------------------------------------------
; Uses InputHook in visible pass-through mode so the user's keystrokes
; reach their target application normally while we capture them.
; ---------------------------------------------------------------------------
class KeyboardRecorder {

    ; --- Properties ---
    session   := ""     ; RecordingSession -- parent session for event dispatch
    inputHook := ""     ; InputHook object
    isActive  := 0      ; 1 = capturing, 0 = stopped

    ; --- Constructor ---
    ; session : RecordingSession -- parent that receives events via AddEvent()
    __New(session) {
        this.session := session

        ; Create InputHook with options:
        ;   V  = Visible: pass-through mode, keystrokes reach the target app
        ;   I1 = Case-sensitive (not strictly needed but preserves intent)
        ;   L0 = No input length limit
        ;   T0 = No timeout
        ih := InputHook("V I1 L0 T0")

        ; Register callbacks using ObjBindMethod so "this" is preserved
        ih.OnKeyDown := ObjBindMethod(this, "_OnKeyDown")
        ih.OnKeyUp   := ObjBindMethod(this, "_OnKeyUp")
        ih.OnChar    := ObjBindMethod(this, "_OnChar")

        ; Enable notifications for ALL keys, including non-text keys
        ; (arrows, function keys, modifiers, etc.)
        ih.KeyOpt("{All}", "N")
        ih.NotifyNonText := true

        this.inputHook := ih
    }

    ; --- Control ---

    ; Start capturing keystrokes
    Start() {
        if (this.isActive)
            return
        try {
            this.inputHook.Start()
            this.isActive := 1
        } catch as err {
            throw RecorderError("Failed to start keyboard recorder: " err.Message, "KeyboardRecorder")
        }
    }

    ; Stop capturing keystrokes
    Stop() {
        if (!this.isActive)
            return
        try {
            this.inputHook.Stop()
            this.isActive := 0
        } catch as err {
            ; Best-effort stop; log but do not throw
            this.isActive := 0
        }
    }

    ; --- Callbacks ---

    ; _OnKeyDown -- Fires for every physical key press
    ; ih : InputHook object (passed by the hook, unused here)
    ; vk : Virtual key code (integer)
    ; sc : Scan code (integer)
    _OnKeyDown(ih, vk, sc) {
        if (!this.isActive || !this.session || this.session.state != "recording")
            return

        keyName := ""
        try {
            keyName := GetKeyName(Format("vk{:x}", vk))
        }
        if (keyName = "")
            keyName := Format("vk{:02x}sc{:03x}", vk, sc)

        event := Map(
            "type", "key_down",
            "time", A_TickCount - this.session.startTime,
            "data", Map(
                "vk", vk,
                "sc", sc,
                "key", keyName
            )
        )
        this.session.AddEvent(event)
    }

    ; _OnKeyUp -- Fires for every physical key release
    _OnKeyUp(ih, vk, sc) {
        if (!this.isActive || !this.session || this.session.state != "recording")
            return

        keyName := ""
        try {
            keyName := GetKeyName(Format("vk{:x}", vk))
        }
        if (keyName = "")
            keyName := Format("vk{:02x}sc{:03x}", vk, sc)

        event := Map(
            "type", "key_up",
            "time", A_TickCount - this.session.startTime,
            "data", Map(
                "vk", vk,
                "sc", sc,
                "key", keyName
            )
        )
        this.session.AddEvent(event)
    }

    ; _OnChar -- Fires when a character is produced (respects dead keys, IME, shift)
    ; ih   : InputHook object
    ; char : The character string produced
    _OnChar(ih, char) {
        if (!this.isActive || !this.session || this.session.state != "recording")
            return

        event := Map(
            "type", "char",
            "time", A_TickCount - this.session.startTime,
            "data", Map(
                "char", char
            )
        )
        this.session.AddEvent(event)
    }
}

; ---------------------------------------------------------------------------
; MouseRecorder -- Captures mouse clicks, movement, and scroll via hotkeys
;                  and SetTimer polling
; ---------------------------------------------------------------------------
; Click hotkeys use the ~ prefix so clicks pass through to the target app.
; Mouse position is polled at a configurable interval, and moves are only
; recorded when the cursor has moved beyond a pixel threshold to avoid
; flooding the event buffer with micro-movements.
; ---------------------------------------------------------------------------
class MouseRecorder {

    ; --- Properties ---
    session       := ""     ; RecordingSession -- parent session
    pollInterval  := 50     ; Mouse position poll rate in ms
    isActive      := 0      ; 1 = capturing, 0 = stopped
    lastX         := 0      ; Previous polled X position
    lastY         := 0      ; Previous polled Y position
    moveThreshold := 3      ; Minimum pixel delta to record a move
    _timerFn      := ""     ; Bound method reference for SetTimer

    ; Bound callback references (stored so we can turn hotkeys off later)
    _fnLButtonDown   := ""
    _fnLButtonUp     := ""
    _fnRButtonDown   := ""
    _fnRButtonUp     := ""
    _fnMButtonDown   := ""
    _fnMButtonUp     := ""
    _fnWheelUp       := ""
    _fnWheelDown     := ""

    ; --- Constructor ---
    ; session : RecordingSession
    ; config  : Config object (optional). If provided, reads poll interval
    ;           and move threshold from it. If omitted, uses defaults.
    __New(session, config := "") {
        this.session := session

        ; Read configuration if a Config object is provided
        if (IsObject(config)) {
            try this.pollInterval  := config.GetInt("recording", "mouse_poll_interval")
            try this.moveThreshold := config.GetInt("recording", "mouse_move_threshold")
        }

        ; Create bound method reference for the position polling timer
        this._timerFn := ObjBindMethod(this, "_PollPosition")

        ; Create bound method references for all hotkey callbacks
        this._fnLButtonDown := ObjBindMethod(this, "_OnClick", "left",   "down")
        this._fnLButtonUp   := ObjBindMethod(this, "_OnClick", "left",   "up")
        this._fnRButtonDown := ObjBindMethod(this, "_OnClick", "right",  "down")
        this._fnRButtonUp   := ObjBindMethod(this, "_OnClick", "right",  "up")
        this._fnMButtonDown := ObjBindMethod(this, "_OnClick", "middle", "down")
        this._fnMButtonUp   := ObjBindMethod(this, "_OnClick", "middle", "up")
        this._fnWheelUp     := ObjBindMethod(this, "_OnWheelUp")
        this._fnWheelDown   := ObjBindMethod(this, "_OnWheelDown")
    }

    ; --- Control ---

    ; Start capturing mouse input
    Start() {
        if (this.isActive)
            return

        try {
            ; Install click hotkeys with ~ prefix for pass-through
            Hotkey "~LButton",    this._fnLButtonDown, "On"
            Hotkey "~LButton Up", this._fnLButtonUp,   "On"
            Hotkey "~RButton",    this._fnRButtonDown, "On"
            Hotkey "~RButton Up", this._fnRButtonUp,   "On"
            Hotkey "~MButton",    this._fnMButtonDown, "On"
            Hotkey "~MButton Up", this._fnMButtonUp,   "On"

            ; Install scroll hotkeys
            Hotkey "~WheelUp",   this._fnWheelUp,   "On"
            Hotkey "~WheelDown", this._fnWheelDown, "On"

            ; Start position polling timer
            SetTimer(this._timerFn, this.pollInterval)

            ; Initialize last known position
            MouseGetPos(&x, &y)
            this.lastX := x
            this.lastY := y

            this.isActive := 1
        } catch as err {
            throw RecorderError("Failed to start mouse recorder: " err.Message, "MouseRecorder")
        }
    }

    ; Stop capturing mouse input
    Stop() {
        if (!this.isActive)
            return

        try {
            ; Turn off all hotkeys
            Hotkey "~LButton",    this._fnLButtonDown, "Off"
            Hotkey "~LButton Up", this._fnLButtonUp,   "Off"
            Hotkey "~RButton",    this._fnRButtonDown, "Off"
            Hotkey "~RButton Up", this._fnRButtonUp,   "Off"
            Hotkey "~MButton",    this._fnMButtonDown, "Off"
            Hotkey "~MButton Up", this._fnMButtonUp,   "Off"
            Hotkey "~WheelUp",   this._fnWheelUp,   "Off"
            Hotkey "~WheelDown", this._fnWheelDown, "Off"
        }

        ; Stop position polling (period=0 disables the timer)
        try SetTimer(this._timerFn, 0)

        this.isActive := 0
    }

    ; --- Callbacks ---

    ; _OnClick -- Handles mouse button down and up events
    ; button    : "left", "right", or "middle"
    ; direction : "down" or "up"
    ; *         : Variadic to accept any extra params from Hotkey callback
    _OnClick(button, direction, *) {
        if (!this.isActive || !this.session || this.session.state != "recording")
            return

        x := 0, y := 0, hwnd := 0
        try MouseGetPos(&x, &y, &hwnd)

        eventType := "mouse_" direction
        event := Map(
            "type", eventType,
            "time", A_TickCount - this.session.startTime,
            "data", Map(
                "button", button,
                "x", x,
                "y", y,
                "hwnd", hwnd
            )
        )
        this.session.AddEvent(event)
    }

    ; _PollPosition -- Called every pollInterval ms by SetTimer
    ; Records a mouse_move event only if the cursor moved beyond the threshold.
    _PollPosition() {
        if (!this.isActive || !this.session || this.session.state != "recording")
            return

        x := 0, y := 0
        try MouseGetPos(&x, &y)

        dx := Abs(x - this.lastX)
        dy := Abs(y - this.lastY)

        if (dx > this.moveThreshold || dy > this.moveThreshold) {
            event := Map(
                "type", "mouse_move",
                "time", A_TickCount - this.session.startTime,
                "data", Map(
                    "x", x,
                    "y", y
                )
            )
            this.session.AddEvent(event)
            this.lastX := x
            this.lastY := y
        }
    }

    ; _OnWheelUp -- Mouse wheel scrolled up
    _OnWheelUp(*) {
        if (!this.isActive || !this.session || this.session.state != "recording")
            return

        x := 0, y := 0
        try MouseGetPos(&x, &y)

        event := Map(
            "type", "mouse_scroll",
            "time", A_TickCount - this.session.startTime,
            "data", Map(
                "direction", "up",
                "x", x,
                "y", y
            )
        )
        this.session.AddEvent(event)
    }

    ; _OnWheelDown -- Mouse wheel scrolled down
    _OnWheelDown(*) {
        if (!this.isActive || !this.session || this.session.state != "recording")
            return

        x := 0, y := 0
        try MouseGetPos(&x, &y)

        event := Map(
            "type", "mouse_scroll",
            "time", A_TickCount - this.session.startTime,
            "data", Map(
                "direction", "down",
                "x", x,
                "y", y
            )
        )
        this.session.AddEvent(event)
    }
}

; ---------------------------------------------------------------------------
; WindowTracker -- Tracks active window changes via SetTimer polling
; ---------------------------------------------------------------------------
; Polls at a configurable interval (default 250ms) and records an event
; only when the active window actually changes (different HWND).
; ---------------------------------------------------------------------------
class WindowTracker {

    ; --- Properties ---
    session      := ""      ; RecordingSession -- parent session
    pollInterval := 250     ; Window check rate in ms
    isActive     := 0       ; 1 = tracking, 0 = stopped
    lastHwnd     := 0       ; HWND of last tracked active window
    lastTitle    := ""      ; Title of last tracked window
    _timerFn     := ""      ; Bound method reference for SetTimer

    ; --- Constructor ---
    ; session : RecordingSession
    ; config  : Config object (optional)
    __New(session, config := "") {
        this.session := session

        ; Read configuration if provided
        if (IsObject(config)) {
            try this.pollInterval := config.GetInt("recording", "window_poll_interval")
        }

        ; Create bound method reference for polling timer
        this._timerFn := ObjBindMethod(this, "_PollActiveWindow")
    }

    ; --- Control ---

    ; Start tracking active window changes
    Start() {
        if (this.isActive)
            return

        try {
            ; Start polling timer
            SetTimer(this._timerFn, this.pollInterval)

            ; Record the initial window state immediately
            this._RecordCurrentWindow()

            this.isActive := 1
        } catch as err {
            throw RecorderError("Failed to start window tracker: " err.Message, "WindowTracker")
        }
    }

    ; Stop tracking active window changes
    Stop() {
        if (!this.isActive)
            return

        ; Stop the polling timer (period=0 disables it)
        try SetTimer(this._timerFn, 0)

        this.isActive := 0
    }

    ; --- Polling ---

    ; _PollActiveWindow -- Called every pollInterval ms by SetTimer
    ; Checks if the active window has changed. If so, records a window_change event.
    _PollActiveWindow() {
        if (!this.isActive || !this.session || this.session.state != "recording")
            return

        hwnd := 0
        try hwnd := WinGetID("A")

        ; Only record if the active window has actually changed
        if (hwnd = this.lastHwnd)
            return

        this._RecordWindowChange(hwnd)
    }

    ; _RecordCurrentWindow -- Captures and records the currently active window
    ; Used during Start() to establish the initial baseline.
    _RecordCurrentWindow() {
        hwnd := 0
        try hwnd := WinGetID("A")
        if (hwnd)
            this._RecordWindowChange(hwnd)
    }

    ; _RecordWindowChange -- Gathers window properties and dispatches event
    ; hwnd : Integer -- the HWND of the new active window
    _RecordWindowChange(hwnd) {
        title := ""
        class := ""
        exe   := ""
        pid   := 0

        try title := WinGetTitle("A")
        try class := WinGetClass("A")
        try exe   := WinGetProcessName("A")
        try pid   := WinGetPID("A")

        event := Map(
            "type", "window_change",
            "time", A_TickCount - this.session.startTime,
            "data", Map(
                "hwnd", hwnd,
                "title", title,
                "class", class,
                "exe", exe,
                "pid", pid
            )
        )
        this.session.AddEvent(event)

        this.lastHwnd  := hwnd
        this.lastTitle := title
    }

    ; --- Query ---

    ; GetCurrentWindow -- Returns a Map of the current active window's properties
    ; Useful for on-demand snapshots (e.g., before UIA capture).
    GetCurrentWindow() {
        hwnd  := 0
        title := ""
        class := ""
        exe   := ""
        pid   := 0

        try hwnd  := WinGetID("A")
        try title := WinGetTitle("A")
        try class := WinGetClass("A")
        try exe   := WinGetProcessName("A")
        try pid   := WinGetPID("A")

        return Map(
            "hwnd", hwnd,
            "title", title,
            "class", class,
            "exe", exe,
            "pid", pid
        )
    }
}

; ---------------------------------------------------------------------------
; RecordingSession -- Orchestrates all three sub-recorders
; ---------------------------------------------------------------------------
; Provides a unified Start/Stop/Pause/Resume interface. All events from
; sub-recorders flow into a single chronological array. An emergency stop
; hotkey (Ctrl+Shift+X) is registered to halt recording immediately.
; ---------------------------------------------------------------------------
class RecordingSession {

    ; --- Properties ---
    state         := "idle"     ; "idle" | "recording" | "paused"
    events        := []         ; Raw event buffer (Array of Maps)
    maxEvents     := 50000      ; Buffer limit
    startTime     := 0          ; A_TickCount at recording start
    pauseTime     := 0          ; A_TickCount when paused (for gap calculation)
    pauseGapTotal := 0          ; Cumulative ms spent paused during this session
    keyboardRec   := ""         ; KeyboardRecorder instance
    mouseRec      := ""         ; MouseRecorder instance
    windowTracker := ""         ; WindowTracker instance
    config        := ""         ; Config object (optional)

    ; Bound callback for the emergency stop hotkey
    _fnEmergencyStop := ""

    ; --- Constructor ---
    ; config : Config object (optional). Reads maxEvents, poll intervals, etc.
    __New(config := "") {
        this.config := config

        ; Read maxEvents from config if available
        if (IsObject(config)) {
            try this.maxEvents := config.GetInt("recording", "max_events")
        }

        ; Initialize sub-recorders, passing this session and config
        this.keyboardRec   := KeyboardRecorder(this)
        this.mouseRec      := MouseRecorder(this, config)
        this.windowTracker := WindowTracker(this, config)

        ; Prepare emergency stop hotkey callback
        this._fnEmergencyStop := ObjBindMethod(this, "_EmergencyStop")
    }

    ; --- Session Control ---

    ; Start -- Begin recording user input
    ; Transitions: idle -> recording, paused -> recording (resume)
    Start() {
        if (this.state = "recording")
            return

        if (this.state = "idle") {
            ; Fresh session: reset everything
            this.events        := []
            this.startTime     := A_TickCount
            this.pauseTime     := 0
            this.pauseGapTotal := 0

            ; Add session start marker
            this.AddEvent(Map(
                "type", "session_start",
                "time", 0,
                "data", Map()
            ))
        } else if (this.state = "paused") {
            ; Resuming from pause: calculate gap and add marker
            pauseDuration := A_TickCount - this.pauseTime
            this.pauseGapTotal += pauseDuration

            this.AddEvent(Map(
                "type", "pause_marker",
                "time", A_TickCount - this.startTime,
                "data", Map(
                    "pause_duration", pauseDuration,
                    "action", "resume"
                )
            ))
        }

        this.state := "recording"

        ; Start all sub-recorders
        try this.keyboardRec.Start()
        try this.mouseRec.Start()
        try this.windowTracker.Start()

        ; Register emergency stop hotkey (Ctrl+Shift+X)
        try Hotkey "^+x", this._fnEmergencyStop, "On"
    }

    ; Stop -- End the recording session and return captured events
    ; Returns: Array -- the complete events array (caller takes ownership)
    Stop() {
        if (this.state = "idle")
            return []

        ; Stop all sub-recorders
        try this.keyboardRec.Stop()
        try this.mouseRec.Stop()
        try this.windowTracker.Stop()

        ; Disable emergency stop hotkey
        try Hotkey "^+x", this._fnEmergencyStop, "Off"

        ; Add session stop marker
        duration := A_TickCount - this.startTime
        eventCount := this.events.Length
        this.AddEvent(Map(
            "type", "session_stop",
            "time", duration,
            "data", Map(
                "event_count", eventCount,
                "duration", duration,
                "pause_total", this.pauseGapTotal
            )
        ))

        ; Capture the events array, then reset
        capturedEvents := this.events
        this.events    := []
        this.state     := "idle"

        return capturedEvents
    }

    ; Pause -- Temporarily suspend recording
    ; Hooks remain installed but events are suppressed (state check in callbacks).
    ; A pause_marker event is added so the Serializer can account for the gap.
    Pause() {
        if (this.state != "recording")
            return

        this.state     := "paused"
        this.pauseTime := A_TickCount

        ; Add pause marker event
        this.AddEvent(Map(
            "type", "pause_marker",
            "time", A_TickCount - this.startTime,
            "data", Map(
                "action", "pause"
            )
        ))
    }

    ; Resume -- Resume recording after a pause (alias for Start when paused)
    Resume() {
        if (this.state != "paused")
            return
        this.Start()
    }

    ; ToggleRecording -- Convenience method for cycling through states
    ; idle -> Start(), recording -> Stop(), paused -> Start() (resume)
    ; Returns: String -- the new state after toggling
    ToggleRecording() {
        switch this.state {
            case "idle":
                this.Start()
            case "recording":
                this.Stop()
            case "paused":
                this.Start()
        }
        return this.state
    }

    ; --- Event Collection ---

    ; AddEvent -- Appends an event Map to the buffer
    ; Enforces the maxEvents limit by dropping the oldest events when exceeded.
    ; event : Map with keys "type", "time", "data"
    AddEvent(event) {
        ; Enforce buffer limit: drop oldest events if we're at capacity
        if (this.events.Length >= this.maxEvents) {
            ; Remove the oldest event (index 1) to make room
            this.events.RemoveAt(1)
        }
        this.events.Push(event)
    }

    ; GetEvents -- Returns a shallow copy of the events buffer (non-destructive)
    GetEvents() {
        copy := []
        copy.Length := this.events.Length
        loop this.events.Length {
            copy[A_Index] := this.events[A_Index]
        }
        return copy
    }

    ; GetEventCount -- Returns the number of events in the buffer
    GetEventCount() {
        return this.events.Length
    }

    ; GetDuration -- Returns elapsed ms since recording started
    ; Accounts for time spent paused if currently paused.
    GetDuration() {
        if (this.state = "idle" || this.startTime = 0)
            return 0

        elapsed := A_TickCount - this.startTime

        ; Subtract total pause time
        totalPause := this.pauseGapTotal
        if (this.state = "paused" && this.pauseTime > 0) {
            ; Currently paused: add the ongoing pause duration
            totalPause += A_TickCount - this.pauseTime
        }

        return elapsed - totalPause
    }

    ; GetState -- Returns the current session state string
    GetState() {
        return this.state
    }

    ; --- Internal ---

    ; _EmergencyStop -- Bound to Ctrl+Shift+X
    ; Immediately stops all recording with no return value processing.
    _EmergencyStop(*) {
        this.Stop()
    }
}
