; ============================================================================
; HttpClient.ahk -- HTTP client module for Claude AHK (AutoHotkey v2.0.21+)
; ============================================================================
;
; Wraps WinHttp.WinHttpRequest.5.1 COM object to provide typed, error-handled
; HTTP communication with the Hercules proxy server.
;
; Features:
;   - GET, POST, PUT, DELETE methods with JSON request/response handling
;   - Bearer token authentication
;   - Configurable timeouts
;   - Retry logic with exponential backoff and jitter
;   - Custom HttpError class with status code and response body
;   - Response object with status, headers, body, and parsed json
;   - HTTPS support (native in WinHttp)
;   - Custom headers per-request
;
; Dependencies:
;   - vendor/Jxon.ahk (JSON parse/stringify)
;
; Usage:
;   #Include "..\vendor\Jxon.ahk"
;   #Include "HttpClient.ahk"
;
;   ; Create client
;   client := HttpClient("https://claude-ahk.hercules.dev")
;   client.SetAuth("cak_tk_your_bearer_token_here")
;   client.SetTimeout(30)
;
;   ; GET request
;   try {
;       response := client.Get("/api/v1/health")
;       MsgBox("Status: " . response["status"])
;       MsgBox("Body: " . response["body"])
;   } catch HttpError as err {
;       MsgBox("Error " . err.statusCode . ": " . err.Message)
;   }
;
;   ; POST request with JSON body
;   try {
;       payload := Map("workflow_id", "wf_abc123", "options", Map("style", "robust"))
;       response := client.Post("/api/v1/generate", payload)
;       data := response["json"]
;       MsgBox("Script: " . data["script"])
;   } catch HttpError as err {
;       if (err.statusCode = 401)
;           MsgBox("Authentication required")
;       else if (err.statusCode = 429)
;           MsgBox("Rate limited, retry after " . err.retryAfter . " seconds")
;       else
;           MsgBox("Error: " . err.Message)
;   }
;
; ============================================================================

#Requires AutoHotkey v2.0


; ============================================================================
; HttpError -- Custom error class for HTTP failures
;
; Properties:
;   statusCode   (Integer)  -- HTTP status code (0 for network/COM errors)
;   responseBody (String)   -- Raw response text from the server
;   url          (String)   -- The full URL that was requested
;   method       (String)   -- HTTP method (GET, POST, PUT, DELETE)
;   isRetryable  (Integer)  -- 1 if the error is transient and retryable
;   retryAfter   (Integer)  -- Seconds to wait before retry (from server), 0 if not set
; ============================================================================
class HttpError extends Error {
    statusCode := 0
    responseBody := ""
    url := ""
    method := ""
    isRetryable := 0
    retryAfter := 0

    /**
     * Construct a new HttpError.
     * @param message     Human-readable error description
     * @param statusCode  HTTP status code (0 for network errors)
     * @param responseBody Raw response text
     * @param url         The request URL
     * @param method      The HTTP method used
     */
    __New(message, statusCode := 0, responseBody := "", url := "", method := "") {
        super.__New(message, -2)
        this.statusCode := statusCode
        this.responseBody := responseBody
        this.url := url
        this.method := method

        ; Determine if this error is retryable
        ; Retryable: network errors (0), 429, 500-599
        ; Not retryable: 400-499 (except 429)
        if (statusCode = 0)
            this.isRetryable := 1
        else if (statusCode = 429)
            this.isRetryable := 1
        else if (statusCode >= 500 && statusCode <= 599)
            this.isRetryable := 1
        else
            this.isRetryable := 0

        ; Try to extract retry_after from response body
        if (statusCode = 429 && responseBody != "") {
            try {
                bodyStr := responseBody
                parsed := Jxon_Load(&bodyStr)
                if (parsed is Map) && parsed.Has("error") && (parsed["error"] is Map) && parsed["error"].Has("retry_after")
                    this.retryAfter := Integer(parsed["error"]["retry_after"])
            }
        }
    }
}


; ============================================================================
; HttpResponse -- Structured response container
;
; This is a plain Map with the following keys:
;   "status"   (Integer) -- HTTP status code
;   "headers"  (Map)     -- Response headers as key-value pairs
;   "body"     (String)  -- Raw response text
;   "json"     (Map|"")  -- Parsed JSON body (empty string if not JSON)
;
; We use a factory function rather than a class to keep things simple.
; ============================================================================
_HttpResponse(status, headers, body, json) {
    resp := Map()
    resp["status"] := status
    resp["headers"] := headers
    resp["body"] := body
    resp["json"] := json
    return resp
}


; ============================================================================
; HttpClient -- Main HTTP client class
;
; Wraps WinHttp.WinHttpRequest.5.1 for communication with the Hercules proxy.
; ============================================================================
class HttpClient {
    ; --- Properties ---
    baseUrl := ""               ; Base URL for all requests (e.g. "https://proxy.hercules.dev")
    authToken := ""             ; Bearer token for Authorization header
    timeout := 30               ; Request timeout in seconds (applied to resolve, connect, send, receive)
    retryCount := 3             ; Maximum number of retry attempts for transient failures
    retryDelay := 1000          ; Base delay in milliseconds between retries
    userAgent := "ClaudeAHK/1.0.0"  ; User-Agent header value
    defaultHeaders := Map()     ; Default headers applied to every request

    ; --- Read-only state ---
    lastStatusCode := 0         ; HTTP status code of the most recent response
    lastError := ""             ; Error message from the most recent failure

    ; --- Private ---
    _whr := ""                  ; WinHttp COM object (created fresh per request for thread safety)

    /**
     * Construct a new HttpClient.
     *
     * @param baseUrl     Base URL for the API server (e.g. "https://claude-ahk.hercules.dev")
     * @param timeout     Request timeout in seconds (default: 30)
     * @param retryCount  Max retry attempts on transient failure (default: 3)
     * @param retryDelay  Base delay in ms between retries (default: 1000)
     */
    __New(baseUrl := "", timeout := 30, retryCount := 3, retryDelay := 1000) {
        this.baseUrl := RTrim(baseUrl, "/")
        this.timeout := timeout
        this.retryCount := retryCount
        this.retryDelay := retryDelay
    }

    ; ====================================================================
    ; Public API Methods
    ; ====================================================================

    /**
     * Send a GET request.
     *
     * @param path    URL path (e.g. "/api/v1/health"). Appended to baseUrl.
     * @param headers (Optional) Map of additional headers for this request.
     * @returns       Map -- HttpResponse with keys: status, headers, body, json
     * @throws        HttpError on failure after retries exhausted
     */
    Get(path, headers := Map()) {
        return this._ExecuteWithRetry("GET", path, "", headers)
    }

    /**
     * Send a POST request with a JSON body.
     *
     * @param path    URL path (e.g. "/api/v1/workflow")
     * @param body    Request body. If Map or Array, auto-serialized to JSON via Jxon_Dump.
     *                If String, sent as-is (caller is responsible for JSON validity).
     * @param headers (Optional) Map of additional headers for this request.
     * @returns       Map -- HttpResponse
     * @throws        HttpError on failure
     */
    Post(path, body := "", headers := Map()) {
        return this._ExecuteWithRetry("POST", path, body, headers)
    }

    /**
     * Send a PUT request with a JSON body.
     *
     * @param path    URL path
     * @param body    Request body (Map/Array auto-serialized, String sent as-is)
     * @param headers (Optional) Map of additional headers
     * @returns       Map -- HttpResponse
     * @throws        HttpError on failure
     */
    Put(path, body := "", headers := Map()) {
        return this._ExecuteWithRetry("PUT", path, body, headers)
    }

    /**
     * Send a DELETE request.
     *
     * @param path    URL path
     * @param headers (Optional) Map of additional headers
     * @returns       Map -- HttpResponse
     * @throws        HttpError on failure
     */
    Delete(path, headers := Map()) {
        return this._ExecuteWithRetry("DELETE", path, "", headers)
    }

    ; ====================================================================
    ; Configuration Methods
    ; ====================================================================

    /**
     * Set the Bearer token for all subsequent requests.
     *
     * @param token  Bearer token string (e.g. "cak_tk_eyJhbGci...")
     *               Pass empty string to clear authentication.
     */
    SetAuth(token) {
        this.authToken := token
    }

    /**
     * Set the request timeout.
     *
     * @param seconds  Timeout in seconds (applied to all phases: resolve, connect, send, receive)
     */
    SetTimeout(seconds) {
        this.timeout := seconds
    }

    /**
     * Set a default header that will be included in every request.
     *
     * @param name   Header name (e.g. "X-Client-Version")
     * @param value  Header value (e.g. "claude-ahk/1.0.0")
     */
    SetDefaultHeader(name, value) {
        this.defaultHeaders[name] := value
    }

    /**
     * Remove a default header.
     *
     * @param name  Header name to remove
     */
    RemoveDefaultHeader(name) {
        if this.defaultHeaders.Has(name)
            this.defaultHeaders.Delete(name)
    }

    ; ====================================================================
    ; Internal: Request Execution with Retry
    ; ====================================================================

    /**
     * Execute an HTTP request with retry logic for transient failures.
     *
     * Retries on:
     *   - Network/COM errors (statusCode = 0)
     *   - HTTP 429 (rate limited)
     *   - HTTP 500-599 (server errors)
     *
     * Does NOT retry on:
     *   - HTTP 400-499 (except 429) -- client errors are permanent
     *
     * Uses exponential backoff with +/- 20% jitter:
     *   delay = retryDelay * (2 ^ attemptIndex) * (0.8 to 1.2)
     *
     * For HTTP 429, uses the server's retry_after value if available.
     *
     * @param method   HTTP method (GET, POST, PUT, DELETE)
     * @param path     URL path
     * @param body     Request body (raw string, or Map/Array to be serialized)
     * @param headers  Map of request-specific headers
     * @returns        Map -- HttpResponse
     * @throws         HttpError after all retries exhausted
     */
    _ExecuteWithRetry(method, path, body, headers) {
        lastErr := ""
        attempts := this.retryCount

        loop attempts {
            attemptIndex := A_Index - 1  ; 0-based for backoff calculation

            try {
                return this._SendRequest(method, path, body, headers)
            } catch HttpError as err {
                lastErr := err
                this.lastError := err.Message

                ; Do not retry non-retryable errors
                if (!err.isRetryable)
                    throw err

                ; Do not retry on last attempt
                if (A_Index >= attempts)
                    throw err

                ; Calculate delay
                if (err.retryAfter > 0) {
                    ; Server told us how long to wait (429 responses)
                    delayMs := err.retryAfter * 1000
                } else {
                    ; Exponential backoff: base * 2^attempt
                    delayMs := this.retryDelay * (2 ** attemptIndex)

                    ; Add jitter: +/- 20%
                    jitterRange := delayMs * 0.4  ; total range is 40% of delay
                    jitter := (Random(0, 1000) / 1000.0) * jitterRange - (jitterRange / 2)
                    delayMs := Integer(delayMs + jitter)

                    ; Clamp to reasonable bounds
                    if (delayMs < 100)
                        delayMs := 100
                    if (delayMs > 30000)
                        delayMs := 30000
                }

                Sleep(delayMs)
            }
        }

        ; Should not reach here, but just in case
        if (lastErr != "")
            throw lastErr
        throw HttpError("Request failed after " . attempts . " attempts", 0, "", this.baseUrl . path, method)
    }

    ; ====================================================================
    ; Internal: Core Request Handler
    ; ====================================================================

    /**
     * Send a single HTTP request (no retries).
     *
     * Steps:
     *   1. Create fresh WinHttp COM object
     *   2. Open connection (synchronous)
     *   3. Set timeouts
     *   4. Set all headers (standard + default + request-specific)
     *   5. Serialize body if needed
     *   6. Send request
     *   7. Read and parse response
     *   8. Check for HTTP errors
     *   9. Return structured response
     *
     * @param method   HTTP method
     * @param path     URL path
     * @param body     Request body
     * @param headers  Request-specific headers
     * @returns        Map -- HttpResponse
     * @throws         HttpError on any failure
     */
    _SendRequest(method, path, body, headers) {
        url := this.baseUrl . path
        bodyStr := ""

        ; Serialize body if it is a Map or Array
        if IsObject(body) && (body is Map || body is Array) {
            bodyStr := Jxon_Dump(body)
        } else if (body is String) {
            bodyStr := body
        }

        ; Determine if we have a request body
        hasBody := (bodyStr != "")

        try {
            ; Create a fresh COM object for each request.
            ; This avoids state leakage between requests and is the recommended
            ; pattern for WinHttp in single-threaded environments.
            whr := ComObject("WinHttp.WinHttpRequest.5.1")

            ; Open connection (synchronous mode = false parameter is actually "async")
            ; false = synchronous. The call to Send() will block until complete.
            whr.Open(method, url, false)

            ; Set timeouts (all in milliseconds):
            ; SetTimeouts(resolveTimeout, connectTimeout, sendTimeout, receiveTimeout)
            timeoutMs := this.timeout * 1000
            whr.SetTimeouts(timeoutMs, timeoutMs, timeoutMs, timeoutMs)

            ; --- Set Headers ---

            ; Standard headers (always sent)
            whr.SetRequestHeader("User-Agent", this.userAgent)
            whr.SetRequestHeader("Accept", "application/json")

            ; Content-Type for requests with a body
            if (hasBody)
                whr.SetRequestHeader("Content-Type", "application/json")

            ; Authorization header
            if (this.authToken != "")
                whr.SetRequestHeader("Authorization", "Bearer " . this.authToken)

            ; Default headers (user-configured, applied to all requests)
            for name, value in this.defaultHeaders {
                whr.SetRequestHeader(name, value)
            }

            ; Request-specific headers (override defaults if same name)
            for name, value in headers {
                whr.SetRequestHeader(name, value)
            }

            ; --- Send Request ---
            if (hasBody)
                whr.Send(bodyStr)
            else
                whr.Send()

            ; --- Read Response ---
            statusCode := whr.Status
            this.lastStatusCode := statusCode
            responseText := ""
            try {
                responseText := whr.ResponseText
            }

            ; --- Parse Response Headers ---
            responseHeaders := this._ParseResponseHeaders(whr)

            ; --- Parse JSON Response ---
            jsonBody := ""
            if (responseText != "") {
                try {
                    tempStr := responseText
                    jsonBody := Jxon_Load(&tempStr)
                }
                ; If JSON parsing fails, jsonBody remains "" -- not all responses are JSON
            }

            ; --- Check for HTTP Errors ---
            if (statusCode >= 400) {
                ; Build error message
                errMsg := "HTTP " . statusCode . " " . method . " " . path

                ; Try to extract a more descriptive message from the JSON error body
                if (jsonBody is Map) && jsonBody.Has("error") && (jsonBody["error"] is Map) {
                    errorObj := jsonBody["error"]
                    if errorObj.Has("message")
                        errMsg := errorObj["message"]
                }

                throw HttpError(errMsg, statusCode, responseText, url, method)
            }

            ; --- Return Structured Response ---
            return _HttpResponse(statusCode, responseHeaders, responseText, jsonBody)

        } catch HttpError {
            ; Re-throw HttpError as-is (do not wrap it)
            throw
        } catch Error as comErr {
            ; COM or network error -- wrap in HttpError with statusCode 0
            this.lastStatusCode := 0
            this.lastError := comErr.Message
            throw HttpError(
                "Network error: " . comErr.Message,
                0,          ; statusCode 0 indicates network/COM failure
                "",         ; no response body
                url,
                method
            )
        }
    }

    ; ====================================================================
    ; Internal: Parse Response Headers
    ; ====================================================================

    /**
     * Parse the raw response headers from WinHttp into a Map.
     *
     * WinHttp.GetAllResponseHeaders() returns a string like:
     *   "Header-Name: value\r\nOther-Header: value\r\n"
     *
     * @param whr  WinHttp COM object (after Send)
     * @returns    Map of header names (lowercase) to values
     */
    _ParseResponseHeaders(whr) {
        hdrs := Map()
        hdrs.CaseSense := false

        try {
            raw := whr.GetAllResponseHeaders()
        } catch {
            return hdrs
        }

        if (raw = "")
            return hdrs

        ; Split by CRLF
        loop parse raw, "`n", "`r" {
            line := Trim(A_LoopField)
            if (line = "")
                continue

            ; Find the first colon
            colonPos := InStr(line, ":")
            if (colonPos > 0) {
                name := Trim(SubStr(line, 1, colonPos - 1))
                value := Trim(SubStr(line, colonPos + 1))
                hdrs[name] := value
            }
        }

        return hdrs
    }
}


; ============================================================================
; Usage Examples (for reference -- not executed when #Included)
; ============================================================================
;
; Example 1: Basic GET request
;
;   client := HttpClient("https://claude-ahk.hercules.dev")
;   client.SetAuth("cak_tk_my_token")
;
;   try {
;       resp := client.Get("/api/v1/health")
;       MsgBox("Server is healthy! Status: " . resp["status"])
;   } catch HttpError as err {
;       MsgBox("Health check failed: " . err.Message)
;   }
;
;
; Example 2: POST workflow submission
;
;   client := HttpClient("https://claude-ahk.hercules.dev")
;   client.SetAuth("cak_tk_my_token")
;   client.SetDefaultHeader("X-Client-Version", "claude-ahk/1.0.0")
;
;   workflow := Map(
;       "name", "Fill expense report",
;       "description", "Automates expense form filling",
;       "actions", [
;           Map("seq", 0, "action_type", "key_combo",
;               "key_combo", Map("modifiers", ["Win"], "key", "r"))
;       ]
;   )
;
;   try {
;       resp := client.Post("/api/v1/workflow", Map("workflow", workflow, "tags", []))
;       data := resp["json"]
;       workflowId := data["workflow_id"]
;       MsgBox("Workflow created: " . workflowId)
;   } catch HttpError as err {
;       if (err.statusCode = 401)
;           MsgBox("Please authenticate first")
;       else if (err.statusCode = 429)
;           MsgBox("Rate limited. Wait " . err.retryAfter . "s and try again.")
;       else
;           MsgBox("Submission failed: " . err.Message)
;   }
;
;
; Example 3: Generate script with error handling
;
;   GenerateScript(client, workflowId) {
;       try {
;           payload := Map(
;               "workflow_id", workflowId,
;               "options", Map(
;                   "style", "robust",
;                   "error_handling", "comprehensive",
;                   "adaptiveness", "high",
;                   "include_comments", 1,
;                   "timing_mode", "natural",
;                   "target_ahk_version", "2.0.21"
;               )
;           )
;           resp := client.Post("/api/v1/generate", payload)
;           data := resp["json"]
;           return Map(
;               "script", data["script"],
;               "explanation", data["explanation"],
;               "confidence", data["confidence"]
;           )
;       } catch HttpError as err {
;           return Map("error", err.Message, "statusCode", err.statusCode)
;       }
;   }
;
;
; Example 4: Refine a script in a session
;
;   RefineScript(client, sessionId, feedback) {
;       path := "/api/v1/session/" . sessionId . "/refine"
;       body := Map(
;           "message", feedback,
;           "options", Map("style", "robust", "include_comments", 1)
;       )
;       try {
;           resp := client.Post(path, body)
;           return resp["json"]
;       } catch HttpError as err {
;           MsgBox("Refinement failed: " . err.Message)
;           return ""
;       }
;   }
;
;
; Example 5: DELETE a session
;
;   try {
;       resp := client.Delete("/api/v1/session/ses_m1n2o3p4")
;       MsgBox("Session deleted. Status: " . resp["status"])
;   } catch HttpError as err {
;       MsgBox("Delete failed: " . err.Message)
;   }
;
;
; Example 6: PUT to update a resource
;
;   try {
;       updates := Map("name", "Updated workflow name", "tags", ["new-tag"])
;       resp := client.Put("/api/v1/workflow/wf_abc123", updates)
;       MsgBox("Updated! Status: " . resp["status"])
;   } catch HttpError as err {
;       MsgBox("Update failed: " . err.Message)
;   }
;
; ============================================================================
