;; ==========================================================================
;; GUI.ahk -- Native Win32 GUI Module for Claude AHK
;; ==========================================================================
;;
;; Provides the graphical user interface for Claude AHK using AHK v2 native
;; Win32 controls (Gui class). This is the MVP implementation -- no WebView2.
;;
;; Layout:
;;   - Toolbar row: Record / Stop / Pause buttons + status label
;;   - Recorded actions: ListView with columns #, Action, Target, Time, Detail
;;   - Generate / Clear buttons
;;   - Generated script: Edit control (monospace, read-only)
;;   - Confidence bar + warnings count
;;   - Execute / Save / Copy / Refine buttons
;;   - StatusBar: server status | action count | session ID
;;
;; Requires: AutoHotkey v2.0.21+ (64-bit)
;; Module:   lib/GUI.ahk
;; Version:  1.0.0
;;
;; Depends:
;;   - lib/Recorder.ahk    (RecordingSession)
;;   - lib/AIEngine.ahk    (AIEngine, ScriptResult)
;;
;; Usage:
;;   #Include "Recorder.ahk"
;;   #Include "AIEngine.ahk"
;;   #Include "GUI.ahk"
;;
;;   recorder := RecordingSession()
;;   engine   := AIEngine(httpClient, engineConfig)
;;   config   := Map("serverUrl", "https://claude-ahk.hercules.dev")
;;
;;   gui := ClaudeAHKGui(recorder, engine, config)
;;   gui.Show()
;;
;; ==========================================================================

#Requires AutoHotkey v2.0


; ============================================================================
; ClaudeAHKGui -- Main application window
; ============================================================================
class ClaudeAHKGui {

    ; --- Dependencies ---
    recorder := ""              ; RecordingSession instance
    aiEngine := ""              ; AIEngine instance
    config   := Map()           ; Configuration map

    ; --- GUI Controls ---
    mainGui       := ""         ; Gui object
    btnRecord     := ""         ; Record button
    btnStop       := ""         ; Stop button
    btnPause      := ""         ; Pause/Resume button
    lblStatus     := ""         ; Status text label
    lvActions     := ""         ; ListView for recorded actions
    btnGenerate   := ""         ; Generate Script button
    btnClear      := ""         ; Clear button
    editScript    := ""         ; Edit control for generated script
    lblScriptHead := ""         ; "Generated Script:" label
    progConfidence := ""        ; Progress bar for confidence score
    lblConfidence := ""         ; Confidence text label
    lblWarnings   := ""         ; Warnings count label
    btnExecute    := ""         ; Execute button
    btnSave       := ""         ; Save button
    btnCopy       := ""         ; Copy button
    btnRefine     := ""         ; Refine button
    statusBar     := ""         ; StatusBar control

    ; --- State ---
    recordingState   := "idle"      ; "idle" | "recording" | "paused"
    currentScript    := ""          ; Currently displayed script text
    currentResult    := ""          ; Current ScriptResult object
    isVisible        := 0           ; Whether window is shown
    _closeCallback   := ""          ; External close callback
    _updateTimerFn   := ""          ; Bound timer function for live updates
    _connTimerFn     := ""          ; Bound timer function for connection check
    _actionRows      := 0           ; Number of rows in the actions ListView
    _hotkeysBound    := 0           ; Whether hotkeys are registered

    ; --- Constants ---
    static APP_TITLE     := "Claude AHK - AI Desktop Automation"
    static MIN_WIDTH     := 640
    static MIN_HEIGHT    := 480
    static DEFAULT_WIDTH := 800
    static DEFAULT_HEIGHT := 620

    ; ====================================================================
    ; Constructor
    ; ====================================================================

    /**
     * Initialize the GUI with references to the recorder and AI engine.
     *
     * @param recorder  RecordingSession instance
     * @param aiEngine  AIEngine instance
     * @param config    Map with optional keys:
     *   "serverUrl"    (String)  Server URL for display
     *   "windowWidth"  (Integer) Initial window width
     *   "windowHeight" (Integer) Initial window height
     */
    __New(recorder, aiEngine, config := Map()) {
        this.recorder := recorder
        this.aiEngine := aiEngine
        this.config   := config

        ; Create bound timer callbacks
        this._updateTimerFn := ObjBindMethod(this, "_OnUpdateTimer")
        this._connTimerFn   := ObjBindMethod(this, "_OnConnectionTimer")

        ; Build the GUI
        this._CreateMainWindow()
        this._CreateToolbar()
        this._CreateActionsPanel()
        this._CreateMiddleButtons()
        this._CreateScriptPanel()
        this._CreateScriptButtons()
        this._CreateStatusBar()
        this._RegisterHotkeys()

        ; Start connection status timer (check every 5 seconds)
        SetTimer(this._connTimerFn, 5000)
    }

    ; ====================================================================
    ; Public Interface
    ; ====================================================================

    /**
     * Show the main window.
     */
    Show() {
        w := this.config.Has("windowWidth")  ? this.config["windowWidth"]  : ClaudeAHKGui.DEFAULT_WIDTH
        h := this.config.Has("windowHeight") ? this.config["windowHeight"] : ClaudeAHKGui.DEFAULT_HEIGHT
        this.mainGui.Show("w" w " h" h)
        this.isVisible := 1
    }

    /**
     * Hide the main window (minimize to tray).
     */
    Hide() {
        this.mainGui.Hide()
        this.isVisible := 0
    }

    /**
     * Clean up and destroy the GUI. Stops all timers and unregisters hotkeys.
     */
    Destroy() {
        ; Stop timers
        SetTimer(this._updateTimerFn, 0)
        SetTimer(this._connTimerFn, 0)

        ; Unregister hotkeys
        this._UnregisterHotkeys()

        ; Destroy window
        try this.mainGui.Destroy()
        this.isVisible := 0
    }

    /**
     * Update the status label with a message.
     *
     * @param text  Status message string
     * @param type  One of "info", "success", "error", "warning"
     */
    UpdateStatus(text, type := "info") {
        if (!this.lblStatus)
            return

        prefix := ""
        switch type {
            case "success": prefix := "[OK] "
            case "error":   prefix := "[ERR] "
            case "warning": prefix := "[!] "
        }
        this.lblStatus.Value := prefix . text
    }

    /**
     * Update the UI to reflect the current recording state.
     *
     * @param state  One of "idle", "recording", "paused"
     */
    SetRecordingState(state) {
        this.recordingState := state

        switch state {
            case "idle":
                this.btnRecord.Enabled := true
                this.btnStop.Enabled   := false
                this.btnPause.Enabled  := false
                this.btnPause.Text     := "Pause"
                this.btnGenerate.Enabled := (this._actionRows > 0)
                this.UpdateStatus("Ready", "info")
                ; Stop the live update timer
                SetTimer(this._updateTimerFn, 0)

            case "recording":
                this.btnRecord.Enabled := false
                this.btnStop.Enabled   := true
                this.btnPause.Enabled  := true
                this.btnPause.Text     := "Pause"
                this.btnGenerate.Enabled := false
                this.UpdateStatus("Recording...", "info")
                ; Start live update timer (100ms interval)
                SetTimer(this._updateTimerFn, 100)

            case "paused":
                this.btnRecord.Enabled := false
                this.btnStop.Enabled   := true
                this.btnPause.Enabled  := true
                this.btnPause.Text     := "Resume"
                this.btnGenerate.Enabled := false
                this.UpdateStatus("Paused", "warning")
                ; Keep timer running at slower rate for count updates
                SetTimer(this._updateTimerFn, 500)
        }

        ; Update status bar
        this._UpdateStatusBar()
    }

    /**
     * Populate the actions ListView with recorded action summaries.
     *
     * @param actions  Array of action Maps. Each Map should have:
     *   "type" (String), "time" (Integer ms), "data" (Map)
     *   These are raw events from RecordingSession.GetEvents().
     */
    DisplayWorkflow(actions) {
        if (!this.lvActions)
            return

        this.lvActions.Delete()
        this._actionRows := 0

        if (!actions || actions.Length = 0)
            return

        ; Disable redraw for performance during bulk insert
        this.lvActions.Opt("-Redraw")

        rowNum := 0
        for _, evt in actions {
            rowNum++
            evtType := evt.Has("type") ? evt["type"] : "unknown"
            evtTime := evt.Has("time") ? evt["time"] : 0
            evtData := evt.Has("data") ? evt["data"] : Map()

            ; Format time as seconds with 1 decimal
            timeFmt := Format("{:.1f}s", evtTime / 1000)

            ; Derive target and detail from event data
            target := ""
            detail := ""
            this._FormatEventRow(evtType, evtData, &target, &detail)

            this.lvActions.Add("", String(rowNum), evtType, target, timeFmt, detail)
        }

        this._actionRows := rowNum

        ; Re-enable redraw and auto-size columns
        this.lvActions.Opt("+Redraw")
        this._AutoSizeColumns()

        ; Enable generate button if we have actions and not recording
        if (this.recordingState = "idle" && rowNum > 0)
            this.btnGenerate.Enabled := true

        ; Update status bar action count
        this._UpdateStatusBar()
    }

    /**
     * Display a generated script in the code editor panel.
     *
     * @param scriptResult  ScriptResult object from AIEngine.GenerateScript()
     */
    DisplayScript(scriptResult) {
        if (!this.editScript || !scriptResult)
            return

        this.currentResult := scriptResult

        ; Display the script code
        scriptCode := ""
        if (scriptResult is ScriptResult) {
            scriptCode := scriptResult.scriptCode
            this.currentScript := scriptCode

            ; Update confidence bar
            confidence := scriptResult.confidenceScore
            pct := Round(confidence * 100)
            this.progConfidence.Value := pct
            this.lblConfidence.Value := "Confidence: " pct "%"

            ; Update warnings count
            warnCount := scriptResult.warnings.Length
            this.lblWarnings.Value := "Warnings: " warnCount
        } else if (scriptResult is String) {
            scriptCode := scriptResult
            this.currentScript := scriptCode
        }

        ; Build header comment block
        header := "; Generated by Claude AHK`r`n"
        if (this.currentResult is ScriptResult) {
            if (this.currentResult.confidenceScore > 0)
                header .= "; Confidence: " Format("{:.2f}", this.currentResult.confidenceScore) "`r`n"
            if (this.currentResult.explanation != "")
                header .= "; " this.currentResult.explanation "`r`n"
        }
        header .= "`r`n"

        this.editScript.Value := header . scriptCode

        ; Enable action buttons
        this.btnExecute.Enabled := (scriptCode != "")
        this.btnSave.Enabled    := (scriptCode != "")
        this.btnCopy.Enabled    := (scriptCode != "")
        this.btnRefine.Enabled  := (scriptCode != "")

        this.UpdateStatus("Script generated successfully", "success")
    }

    /**
     * Register a callback to be invoked when the window is closed.
     *
     * @param callback  Function object taking no arguments
     */
    OnClose(callback) {
        this._closeCallback := callback
    }

    ; ====================================================================
    ; GUI Construction (Private)
    ; ====================================================================

    /**
     * Create the main window with resize support.
     */
    _CreateMainWindow() {
        minW := ClaudeAHKGui.MIN_WIDTH
        minH := ClaudeAHKGui.MIN_HEIGHT
        this.mainGui := Gui("+Resize +MinSize" minW "x" minH, ClaudeAHKGui.APP_TITLE)
        this.mainGui.BackColor := "F0F0F0"

        ; Register window events
        this.mainGui.OnEvent("Close", ObjBindMethod(this, "_OnGuiClose"))
        this.mainGui.OnEvent("Size", ObjBindMethod(this, "_OnGuiSize"))
    }

    /**
     * Create the toolbar row: Record, Stop, Pause buttons and status label.
     */
    _CreateToolbar() {
        g := this.mainGui

        ; Set default font for toolbar
        g.SetFont("s9", "Segoe UI")

        ; Record button (green-tinted via styling later)
        this.btnRecord := g.Add("Button", "x10 y8 w80 h30", "Record")
        this.btnRecord.OnEvent("Click", ObjBindMethod(this, "_OnRecord"))

        ; Stop button
        this.btnStop := g.Add("Button", "x95 y8 w80 h30", "Stop")
        this.btnStop.OnEvent("Click", ObjBindMethod(this, "_OnStop"))
        this.btnStop.Enabled := false

        ; Pause button
        this.btnPause := g.Add("Button", "x180 y8 w80 h30", "Pause")
        this.btnPause.OnEvent("Click", ObjBindMethod(this, "_OnPause"))
        this.btnPause.Enabled := false

        ; Separator (vertical bar visual via a thin text)
        g.Add("Text", "x270 y8 w1 h30 0x11")  ; SS_ETCHEDVERT

        ; Settings button (gear icon area)
        btnSettings := g.Add("Button", "x280 y8 w70 h30", "Settings")
        btnSettings.OnEvent("Click", ObjBindMethod(this, "_OnSettings"))

        ; Status label (right-aligned in toolbar area)
        this.lblStatus := g.Add("Text", "x360 y14 w420 h20 Right", "Status: Ready")
        this.lblStatus.SetFont("s9", "Segoe UI")
    }

    /**
     * Create the recorded actions panel: label + ListView.
     */
    _CreateActionsPanel() {
        g := this.mainGui

        ; Section label
        g.SetFont("s9 Bold", "Segoe UI")
        g.Add("Text", "x10 y46 w200 h18", "Recorded Actions:")
        g.SetFont("s9 Norm", "Segoe UI")

        ; ListView for actions
        this.lvActions := g.Add("ListView",
            "x10 y66 w780 h160 +Grid +LV0x4000 +NoSortHdr",
            ["#", "Action", "Target", "Time", "Detail"])

        ; Set initial column widths
        this.lvActions.ModifyCol(1, 40)    ; #
        this.lvActions.ModifyCol(2, 100)   ; Action
        this.lvActions.ModifyCol(3, 250)   ; Target
        this.lvActions.ModifyCol(4, 70)    ; Time
        this.lvActions.ModifyCol(5, 300)   ; Detail
    }

    /**
     * Create the middle button row: Generate Script and Clear.
     */
    _CreateMiddleButtons() {
        g := this.mainGui

        g.SetFont("s9", "Segoe UI")

        ; Generate Script button
        this.btnGenerate := g.Add("Button", "x10 y232 w120 h28", "Generate Script")
        this.btnGenerate.OnEvent("Click", ObjBindMethod(this, "_OnGenerate"))
        this.btnGenerate.Enabled := false

        ; Clear button
        this.btnClear := g.Add("Button", "x140 y232 w70 h28", "Clear")
        this.btnClear.OnEvent("Click", ObjBindMethod(this, "_OnClear"))
    }

    /**
     * Create the generated script panel: label + code editor + confidence bar.
     */
    _CreateScriptPanel() {
        g := this.mainGui

        ; Section label
        g.SetFont("s9 Bold", "Segoe UI")
        this.lblScriptHead := g.Add("Text", "x10 y266 w200 h18", "Generated Script:")
        g.SetFont("s9 Norm", "Segoe UI")

        ; Code editor (monospace, read-only, multi-line)
        g.SetFont("s10", "Consolas")
        this.editScript := g.Add("Edit",
            "x10 y286 w780 h180 ReadOnly +Multi +WantTab +HScroll VScroll",
            "")
        g.SetFont("s9", "Segoe UI")

        ; Confidence progress bar
        this.progConfidence := g.Add("Progress", "x10 y472 w200 h18 Range0-100", 0)

        ; Confidence label
        this.lblConfidence := g.Add("Text", "x215 y473 w130 h18", "Confidence: --%")

        ; Warnings label
        this.lblWarnings := g.Add("Text", "x350 y473 w120 h18", "Warnings: 0")
    }

    /**
     * Create the script action buttons: Execute, Save, Copy, Refine.
     */
    _CreateScriptButtons() {
        g := this.mainGui

        g.SetFont("s9", "Segoe UI")

        ; Execute button
        this.btnExecute := g.Add("Button", "x10 y496 w80 h28", "Execute")
        this.btnExecute.OnEvent("Click", ObjBindMethod(this, "_OnExecute"))
        this.btnExecute.Enabled := false

        ; Save button
        this.btnSave := g.Add("Button", "x96 y496 w80 h28", "Save")
        this.btnSave.OnEvent("Click", ObjBindMethod(this, "_OnSave"))
        this.btnSave.Enabled := false

        ; Copy button
        this.btnCopy := g.Add("Button", "x182 y496 w80 h28", "Copy")
        this.btnCopy.OnEvent("Click", ObjBindMethod(this, "_OnCopy"))
        this.btnCopy.Enabled := false

        ; Refine button
        this.btnRefine := g.Add("Button", "x268 y496 w80 h28", "Refine")
        this.btnRefine.OnEvent("Click", ObjBindMethod(this, "_OnRefine"))
        this.btnRefine.Enabled := false
    }

    /**
     * Create the status bar with three sections.
     */
    _CreateStatusBar() {
        this.statusBar := this.mainGui.Add("StatusBar",, "Server: Checking...")
        this.statusBar.SetParts(250, 150)  ; 3 sections
        this.statusBar.SetText("Server: Checking...", 1)
        this.statusBar.SetText("Actions: 0", 2)
        this.statusBar.SetText("Session: (none)", 3)
    }

    ; ====================================================================
    ; Hotkey Registration
    ; ====================================================================

    /**
     * Register application-level hotkeys for recording and script operations.
     */
    _RegisterHotkeys() {
        if (this._hotkeysBound)
            return

        try {
            ; F5 or Ctrl+R: Start recording
            Hotkey "F5", ObjBindMethod(this, "_OnHotkeyRecord"), "On"
            Hotkey "^r", ObjBindMethod(this, "_OnHotkeyRecord"), "On"

            ; F6 or Escape: Stop recording
            Hotkey "F6", ObjBindMethod(this, "_OnHotkeyStop"), "On"

            ; F7: Pause/Resume
            Hotkey "F7", ObjBindMethod(this, "_OnHotkeyPause"), "On"

            ; Ctrl+G: Generate script
            Hotkey "^g", ObjBindMethod(this, "_OnHotkeyGenerate"), "On"

            ; Ctrl+E: Execute script
            Hotkey "^e", ObjBindMethod(this, "_OnHotkeyExecute"), "On"

            this._hotkeysBound := 1
        } catch as err {
            ; Non-fatal: hotkeys are convenience, not required
        }
    }

    /**
     * Unregister all application-level hotkeys.
     */
    _UnregisterHotkeys() {
        if (!this._hotkeysBound)
            return

        try Hotkey "F5", "Off"
        try Hotkey "^r", "Off"
        try Hotkey "F6", "Off"
        try Hotkey "F7", "Off"
        try Hotkey "^g", "Off"
        try Hotkey "^e", "Off"

        this._hotkeysBound := 0
    }

    ; ====================================================================
    ; Hotkey Handlers (thin wrappers that accept the * variadic param)
    ; ====================================================================

    _OnHotkeyRecord(*) {
        this._OnRecord()
    }

    _OnHotkeyStop(*) {
        this._OnStop()
    }

    _OnHotkeyPause(*) {
        this._OnPause()
    }

    _OnHotkeyGenerate(*) {
        this._OnGenerate()
    }

    _OnHotkeyExecute(*) {
        this._OnExecute()
    }

    ; ====================================================================
    ; Event Handlers -- Toolbar Buttons
    ; ====================================================================

    /**
     * Start recording. Clears previous actions and begins a new session.
     */
    _OnRecord(*) {
        if (!this.recorder || this.recordingState = "recording")
            return

        try {
            ; Clear previous workflow display if starting fresh
            if (this.recordingState = "idle") {
                this.lvActions.Delete()
                this._actionRows := 0
            }

            this.recorder.Start()
            this.SetRecordingState("recording")
        } catch as err {
            this.UpdateStatus("Failed to start recording: " err.Message, "error")
        }
    }

    /**
     * Stop recording and display captured actions.
     */
    _OnStop(*) {
        if (!this.recorder || this.recordingState = "idle")
            return

        try {
            events := this.recorder.Stop()
            this.SetRecordingState("idle")

            ; Filter out internal marker events for display
            displayEvents := this._FilterDisplayEvents(events)
            this.DisplayWorkflow(displayEvents)

            count := displayEvents.Length
            this.UpdateStatus("Recorded " count " actions", "success")
        } catch as err {
            this.SetRecordingState("idle")
            this.UpdateStatus("Error stopping recording: " err.Message, "error")
        }
    }

    /**
     * Toggle pause/resume during recording.
     */
    _OnPause(*) {
        if (!this.recorder)
            return

        if (this.recordingState = "recording") {
            try {
                this.recorder.Pause()
                this.SetRecordingState("paused")
            } catch as err {
                this.UpdateStatus("Failed to pause: " err.Message, "error")
            }
        } else if (this.recordingState = "paused") {
            try {
                this.recorder.Resume()
                this.SetRecordingState("recording")
            } catch as err {
                this.UpdateStatus("Failed to resume: " err.Message, "error")
            }
        }
    }

    ; ====================================================================
    ; Event Handlers -- Generate / Clear
    ; ====================================================================

    /**
     * Serialize the recorded workflow and send to AIEngine for script generation.
     */
    _OnGenerate(*) {
        if (!this.aiEngine || !this.recorder)
            return

        ; Ensure we have recorded events
        events := this.recorder.GetEvents()
        if (!events || events.Length = 0) {
            this.UpdateStatus("No recorded actions to generate from", "warning")
            return
        }

        ; Disable generate button during processing
        this.btnGenerate.Enabled := false
        this.UpdateStatus("Generating script... (this may take a moment)", "info")

        try {
            ; Build a minimal workflow map from the recorded events.
            ; In a full implementation, the Serializer module would handle this.
            ; Here we build a basic structure that AIEngine.GenerateScript() expects.
            workflow := Map(
                "actions", events,
                "metadata", Map(
                    "event_count", events.Length,
                    "duration_ms", this.recorder.GetDuration()
                )
            )

            result := this.aiEngine.GenerateScript(workflow)
            this.DisplayScript(result)

        } catch as err {
            this.UpdateStatus("Generation failed: " err.Message, "error")
            this.btnGenerate.Enabled := true
        }
    }

    /**
     * Clear all recorded actions and the generated script display.
     */
    _OnClear(*) {
        ; Clear actions list
        this.lvActions.Delete()
        this._actionRows := 0

        ; Clear script display
        this.editScript.Value := ""
        this.currentScript := ""
        this.currentResult := ""

        ; Reset confidence
        this.progConfidence.Value := 0
        this.lblConfidence.Value := "Confidence: --%"
        this.lblWarnings.Value := "Warnings: 0"

        ; Disable script action buttons
        this.btnExecute.Enabled  := false
        this.btnSave.Enabled     := false
        this.btnCopy.Enabled     := false
        this.btnRefine.Enabled   := false
        this.btnGenerate.Enabled := false

        this.UpdateStatus("Cleared", "info")
        this._UpdateStatusBar()
    }

    ; ====================================================================
    ; Event Handlers -- Script Action Buttons
    ; ====================================================================

    /**
     * Execute the currently displayed script via AIEngine.
     */
    _OnExecute(*) {
        if (!this.aiEngine || this.currentScript = "")
            return

        ; Confirm execution with the user
        result := MsgBox(
            "Execute the generated script?`n`nThe script will run in a new AHK process.",
            "Confirm Execution",
            "OKCancel Icon! Default2"
        )

        if (result != "OK")
            return

        this.UpdateStatus("Executing script...", "info")
        this.btnExecute.Enabled := false

        try {
            execResult := this.aiEngine.ExecuteScript(this.currentScript)

            if (execResult.success) {
                this.UpdateStatus("Script executed successfully (exit code: " execResult.exitCode
                    ", " execResult.durationMs "ms)", "success")
            } else if (execResult.timedOut) {
                this.UpdateStatus("Script execution timed out after "
                    this.aiEngine.executionTimeout "s", "warning")
            } else {
                errMsg := execResult.stderr != "" ? execResult.stderr : "Exit code " execResult.exitCode
                this.UpdateStatus("Script execution failed: " errMsg, "error")
            }
        } catch as err {
            this.UpdateStatus("Execution error: " err.Message, "error")
        } finally {
            this.btnExecute.Enabled := true
        }
    }

    /**
     * Save the generated script to disk via AIEngine.SaveScript().
     */
    _OnSave(*) {
        if (!this.aiEngine || this.currentScript = "")
            return

        ; Prompt for a script name
        saveName := InputBox(
            "Enter a name for the saved script:",
            "Save Script",
            "w350 h120",
            "generated-" FormatTime(, "yyyyMMdd-HHmmss")
        )

        if (saveName.Result != "OK" || saveName.Value = "")
            return

        try {
            ; Build metadata from current result
            metadata := Map()
            if (this.currentResult is ScriptResult) {
                metadata["explanation"] := this.currentResult.explanation
                metadata["confidenceScore"] := this.currentResult.confidenceScore
                metadata["warnings"] := this.currentResult.warnings
                metadata["savedAt"] := FormatTime(, "yyyy-MM-dd HH:mm:ss")
            }

            paths := this.aiEngine.SaveScript(this.currentScript, saveName.Value, metadata)
            scriptPath := paths.Has("scriptPath") ? paths["scriptPath"] : "(unknown)"
            this.UpdateStatus("Script saved: " scriptPath, "success")
        } catch as err {
            this.UpdateStatus("Save failed: " err.Message, "error")
        }
    }

    /**
     * Copy the generated script to the clipboard.
     */
    _OnCopy(*) {
        if (this.currentScript = "")
            return

        A_Clipboard := this.currentScript
        this.UpdateStatus("Script copied to clipboard", "success")
    }

    /**
     * Open a refinement dialog to send feedback and regenerate the script.
     */
    _OnRefine(*) {
        if (!this.aiEngine || !this.currentResult)
            return

        ; Get the session ID for refinement context
        sessionId := ""
        if (this.currentResult is ScriptResult)
            sessionId := this.currentResult.sessionId

        if (sessionId = "") {
            this.UpdateStatus("No session context for refinement", "warning")
            return
        }

        ; Prompt the user for feedback
        feedback := InputBox(
            "Describe what should be changed or fixed:`n"
            "(e.g., 'Add error handling for missing window', 'Use Tab key instead of clicking')",
            "Refine Script",
            "w450 h160",
            ""
        )

        if (feedback.Result != "OK" || feedback.Value = "")
            return

        this.UpdateStatus("Refining script...", "info")
        this.btnRefine.Enabled := false

        try {
            refinedResult := this.aiEngine.RefineScript(
                this.currentScript,
                feedback.Value,
                sessionId
            )
            this.DisplayScript(refinedResult)
            this.UpdateStatus("Script refined successfully", "success")
        } catch as err {
            this.UpdateStatus("Refinement failed: " err.Message, "error")
        } finally {
            this.btnRefine.Enabled := true
        }
    }

    ; ====================================================================
    ; Settings Dialog
    ; ====================================================================

    /**
     * Open a settings dialog for server URL and basic options.
     */
    _OnSettings(*) {
        settingsGui := Gui("+Owner" this.mainGui.Hwnd " +ToolWindow", "Settings")
        settingsGui.SetFont("s9", "Segoe UI")

        ; Server URL
        settingsGui.Add("Text", "x15 y15 w100", "Server URL:")
        currentUrl := this.config.Has("serverUrl") ? this.config["serverUrl"]
            : "https://claude-ahk.hercules.dev"
        edtUrl := settingsGui.Add("Edit", "x15 y35 w350 h24", currentUrl)

        ; Execution timeout
        settingsGui.Add("Text", "x15 y70 w150", "Execution Timeout (sec):")
        currentTimeout := (this.aiEngine && this.aiEngine.executionTimeout)
            ? String(this.aiEngine.executionTimeout) : "60"
        edtTimeout := settingsGui.Add("Edit", "x15 y90 w80 h24 Number", currentTimeout)

        ; Auto-refine toggle
        autoRefine := (this.aiEngine && this.aiEngine.autoRefine) ? 1 : 0
        chkAutoRefine := settingsGui.Add("Checkbox", "x15 y125 w250 h20 Checked" autoRefine,
            "Auto-refine on execution failure")

        ; Hotkeys section
        settingsGui.Add("GroupBox", "x15 y155 w350 h130", "Hotkeys")
        settingsGui.Add("Text", "x25 y175 w130", "Start Recording:")
        settingsGui.Add("Text", "x160 y175 w150", "F5 or Ctrl+R")
        settingsGui.Add("Text", "x25 y195 w130", "Stop Recording:")
        settingsGui.Add("Text", "x160 y195 w150", "F6")
        settingsGui.Add("Text", "x25 y215 w130", "Pause/Resume:")
        settingsGui.Add("Text", "x160 y215 w150", "F7")
        settingsGui.Add("Text", "x25 y235 w130", "Generate Script:")
        settingsGui.Add("Text", "x160 y235 w150", "Ctrl+G")
        settingsGui.Add("Text", "x25 y255 w130", "Execute Script:")
        settingsGui.Add("Text", "x160 y255 w150", "Ctrl+E")

        ; OK / Cancel buttons
        btnOK := settingsGui.Add("Button", "x195 y300 w80 h30 Default", "OK")
        btnCancel := settingsGui.Add("Button", "x285 y300 w80 h30", "Cancel")

        ; OK handler: apply settings
        btnOK.OnEvent("Click", (*) => this._ApplySettings(
            settingsGui, edtUrl.Value, edtTimeout.Value, chkAutoRefine.Value))

        ; Cancel handler: just close
        btnCancel.OnEvent("Click", (*) => settingsGui.Destroy())
        settingsGui.OnEvent("Escape", (*) => settingsGui.Destroy())

        settingsGui.Show("w385 h345")
    }

    /**
     * Apply settings from the settings dialog.
     */
    _ApplySettings(settingsGui, serverUrl, timeout, autoRefine) {
        ; Validate server URL
        if (serverUrl != "" && SubStr(serverUrl, 1, 8) != "https://" && SubStr(serverUrl, 1, 7) != "http://") {
            MsgBox("Server URL must start with https:// or http://", "Validation Error", "Icon!")
            return
        }

        ; Apply server URL
        if (serverUrl != "")
            this.config["serverUrl"] := serverUrl

        ; Apply execution timeout
        if (timeout != "" && this.aiEngine) {
            try this.aiEngine.executionTimeout := Integer(timeout)
        }

        ; Apply auto-refine
        if (this.aiEngine)
            this.aiEngine.autoRefine := !!autoRefine

        settingsGui.Destroy()
        this.UpdateStatus("Settings updated", "success")
    }

    ; ====================================================================
    ; Timer Callbacks (Private)
    ; ====================================================================

    /**
     * Live update timer (100ms during recording).
     * Updates action count and duration in the status bar.
     */
    _OnUpdateTimer() {
        if (!this.recorder)
            return

        count := this.recorder.GetEventCount()
        duration := this.recorder.GetDuration()
        durationSec := Format("{:.1f}", duration / 1000)

        this.statusBar.SetText("Actions: " count, 2)

        ; Update status label with live recording info
        if (this.recordingState = "recording") {
            this.lblStatus.Value := "Recording... " count " events (" durationSec "s)"
        } else if (this.recordingState = "paused") {
            this.lblStatus.Value := "Paused | " count " events (" durationSec "s)"
        }
    }

    /**
     * Connection check timer (5000ms).
     * Pings the server and updates the status bar connection indicator.
     */
    _OnConnectionTimer() {
        ; Only check if we have an AI engine with an HTTP client
        if (!this.aiEngine || !this.aiEngine.httpClient) {
            this.statusBar.SetText("Server: No client", 1)
            return
        }

        try {
            ; Attempt a lightweight health check
            response := this.aiEngine.httpClient.Get(this.aiEngine.serverBasePath . "/health")
            this.statusBar.SetText("Server: Connected", 1)
        } catch {
            this.statusBar.SetText("Server: Disconnected", 1)
        }
    }

    ; ====================================================================
    ; GUI Events (Private)
    ; ====================================================================

    /**
     * Handle window close event. Minimizes to tray or invokes the close callback.
     */
    _OnGuiClose(*) {
        if (this._closeCallback) {
            this._closeCallback.Call()
        } else {
            this.Hide()
        }
        return true  ; Prevent default close behavior
    }

    /**
     * Handle window resize. Repositions and resizes controls to fill the window.
     */
    _OnGuiSize(thisGui, minMax, width, height) {
        if (minMax = -1)  ; Minimized -- do nothing
            return

        ; Margins
        mx := 10
        sbHeight := 22  ; Approximate StatusBar height

        ; Available width for controls
        cw := width - (mx * 2)

        ; Toolbar: status label stretches to fill remaining width
        try this.lblStatus.Move(360, 14, cw - 350, 20)

        ; Actions ListView: full width, fixed height
        try this.lvActions.Move(mx, 66, cw, 160)

        ; Middle buttons stay at fixed Y
        ; (btnGenerate and btnClear positions are fine at x10, x140)

        ; Script label
        try this.lblScriptHead.Move(mx, 266, 200, 18)

        ; Script editor: stretch both horizontally and vertically
        scriptTop := 286
        scriptButtonRowH := 36
        confidenceRowH := 22
        bottomPadding := sbHeight + scriptButtonRowH + confidenceRowH + 30
        scriptH := height - scriptTop - bottomPadding
        if (scriptH < 60)
            scriptH := 60
        try this.editScript.Move(mx, scriptTop, cw, scriptH)

        ; Confidence row: below script editor
        confY := scriptTop + scriptH + 4
        try this.progConfidence.Move(mx, confY, 200, 18)
        try this.lblConfidence.Move(215, confY + 1, 130, 18)
        try this.lblWarnings.Move(350, confY + 1, 120, 18)

        ; Script action buttons: below confidence row
        btnY := confY + 24
        try this.btnExecute.Move(mx, btnY, 80, 28)
        try this.btnSave.Move(mx + 86, btnY, 80, 28)
        try this.btnCopy.Move(mx + 172, btnY, 80, 28)
        try this.btnRefine.Move(mx + 258, btnY, 80, 28)
    }

    ; ====================================================================
    ; Status Bar Helpers (Private)
    ; ====================================================================

    /**
     * Update all three status bar sections with current state.
     */
    _UpdateStatusBar() {
        ; Section 2: Action count
        count := this._actionRows
        if (this.recorder && this.recordingState != "idle") {
            count := this.recorder.GetEventCount()
        }
        this.statusBar.SetText("Actions: " count, 2)

        ; Section 3: Session ID
        sessionId := "(none)"
        if (this.currentResult is ScriptResult && this.currentResult.sessionId != "") {
            sid := this.currentResult.sessionId
            ; Truncate for display
            if (StrLen(sid) > 12)
                sid := SubStr(sid, 1, 12) "..."
            sessionId := sid
        }
        this.statusBar.SetText("Session: " sessionId, 3)
    }

    ; ====================================================================
    ; Event Formatting Helpers (Private)
    ; ====================================================================

    /**
     * Format a raw event into target and detail strings for ListView display.
     *
     * @param evtType   Event type string (e.g. "key_down", "mouse_down")
     * @param evtData   Event data Map
     * @param &target   Output: target description
     * @param &detail   Output: detail description
     */
    _FormatEventRow(evtType, evtData, &target, &detail) {
        target := "-"
        detail := ""

        switch evtType {
            case "key_down", "key_up":
                keyName := evtData.Has("key") ? evtData["key"] : "?"
                target := keyName
                detail := evtType = "key_down" ? "pressed" : "released"

            case "char":
                ch := evtData.Has("char") ? evtData["char"] : "?"
                target := "Character"
                detail := "`"" ch "`""

            case "mouse_down", "mouse_up":
                btn := evtData.Has("button") ? evtData["button"] : "?"
                x   := evtData.Has("x") ? evtData["x"] : 0
                y   := evtData.Has("y") ? evtData["y"] : 0
                target := btn " button"
                action := evtType = "mouse_down" ? "down" : "up"
                detail := action " at (" x ", " y ")"

            case "mouse_move":
                x := evtData.Has("x") ? evtData["x"] : 0
                y := evtData.Has("y") ? evtData["y"] : 0
                target := "Cursor"
                detail := "moved to (" x ", " y ")"

            case "mouse_scroll":
                dir := evtData.Has("direction") ? evtData["direction"] : "?"
                x   := evtData.Has("x") ? evtData["x"] : 0
                y   := evtData.Has("y") ? evtData["y"] : 0
                target := "Scroll"
                detail := dir " at (" x ", " y ")"

            case "window_change":
                title := evtData.Has("title") ? evtData["title"] : "(unknown)"
                exe   := evtData.Has("exe")   ? evtData["exe"]   : ""
                ; Truncate long titles
                if (StrLen(title) > 40)
                    title := SubStr(title, 1, 37) "..."
                target := title
                detail := exe

            case "session_start":
                target := "Session"
                detail := "Recording started"

            case "session_stop":
                target := "Session"
                ec := evtData.Has("event_count") ? evtData["event_count"] : 0
                dur := evtData.Has("duration") ? evtData["duration"] : 0
                detail := "Stopped (" ec " events, " Format("{:.1f}", dur / 1000) "s)"

            case "pause_marker":
                target := "Session"
                action := evtData.Has("action") ? evtData["action"] : "?"
                detail := action = "pause" ? "Paused" : "Resumed"

            default:
                target := evtType
                detail := "(raw event)"
        }
    }

    /**
     * Filter out internal/noise events for display. Keeps only events that
     * are meaningful to the user.
     *
     * @param events  Raw events array from RecordingSession.Stop()
     * @returns Array of filtered events
     */
    _FilterDisplayEvents(events) {
        filtered := []

        for _, evt in events {
            evtType := evt.Has("type") ? evt["type"] : ""

            ; Skip high-frequency mouse_move events for display clarity.
            ; Keep clicks, keystrokes, window changes, and session markers.
            switch evtType {
                case "mouse_move":
                    continue  ; Skip mouse moves (too noisy for list view)
                case "key_up":
                    continue  ; Skip key-up (key_down + char are sufficient)
                default:
                    filtered.Push(evt)
            }
        }

        return filtered
    }

    /**
     * Auto-size ListView columns based on content.
     */
    _AutoSizeColumns() {
        ; Column 1 (#): fixed narrow
        this.lvActions.ModifyCol(1, 40)
        ; Columns 2-5: auto-size to content with minimum widths
        this.lvActions.ModifyCol(2, "AutoHdr")
        this.lvActions.ModifyCol(3, "AutoHdr")
        this.lvActions.ModifyCol(4, 70)
        this.lvActions.ModifyCol(5, "AutoHdr")
    }
}
