; ============================================================================
; AIEngine.ahk -- AI Script Generation and Execution Engine for Claude AHK
; ============================================================================
;
; The central orchestration module that ties together:
;   - HttpClient.ahk  (sends requests to Hercules proxy)
;   - Serializer.ahk  (converts recordings to JSON)
;   - Claude API responses (receives generated AHK scripts)
;
; Manages the full lifecycle: serialize -> submit -> validate -> execute -> refine.
;
; Requires: AutoHotkey v2.0.21+ (64-bit)
; Depends:
;   - vendor/Jxon.ahk     (JSON parse/stringify)
;   - lib/HttpClient.ahk   (HTTP client for server communication)
;   - lib/Serializer.ahk   (workflow serialization -- data passed in, not imported)
;
; Classes:
;   AIEngine         -- Main engine class (public API)
;   ScriptResult     -- Structured result from script generation
;   ExecutionResult  -- Structured result from script execution
;   AIEngineError    -- Custom error type for this module
;
; Usage:
;   #Include "..\vendor\Jxon.ahk"
;   #Include "HttpClient.ahk"
;   #Include "AIEngine.ahk"
;
;   client := HttpClient("https://claude-ahk.hercules.dev")
;   client.SetAuth("cak_tk_your_token")
;
;   config := Map(
;       "executionTimeout", 60,
;       "maxRetries", 4,
;       "autoRefine", true,
;       "scriptSaveDir", A_ScriptDir "\scripts",
;       "ahkExePath", A_ScriptDir "\AutoHotkey64.exe"
;   )
;   engine := AIEngine(client, config)
;
;   ; Generate a script from a serialized workflow Map
;   result := engine.GenerateScript(workflowMap)
;   if (result.scriptCode != "") {
;       MsgBox("Script generated with confidence: " . result.confidenceScore)
;       execResult := engine.ExecuteScript(result.scriptCode)
;   }
;
; ============================================================================

#Requires AutoHotkey v2.0


; ============================================================================
; AIEngineError -- Custom error class for the AI Engine module
; ============================================================================
class AIEngineError extends Error {
    category := ""
    retryable := 0

    /**
     * @param message   Human-readable error description
     * @param category  Error category: "network", "validation", "execution", "parse", "safety"
     * @param retryable Whether this error is transient and retryable (1 or 0)
     */
    __New(message, category := "general", retryable := 0) {
        super.__New(message, -2)
        this.category := category
        this.retryable := retryable
    }
}


; ============================================================================
; ExecutionResult -- Structured result from script execution
; ============================================================================
class ExecutionResult {
    success := 0            ; Boolean: 1 if script exited cleanly with code 0
    exitCode := -1          ; Process exit code (-1 = not yet executed)
    stdout := ""            ; Captured standard output from the script
    stderr := ""            ; Captured standard error from the script
    durationMs := 0         ; How long execution took in milliseconds
    timedOut := 0           ; Boolean: 1 if the process was killed due to timeout
    scriptPath := ""        ; Path to the temp script file that was executed
    pid := 0                ; Process ID of the executed script

    __New() {
    }
}


; ============================================================================
; ScriptResult -- Structured result from script generation
; ============================================================================
class ScriptResult {
    scriptId := ""          ; Server-assigned generation ID (e.g. "gen_x7y8z9w0")
    scriptCode := ""        ; The generated AHK v2 script text
    explanation := ""       ; Human-readable explanation of what the script does
    confidenceScore := 0.0  ; 0.0-1.0 confidence that the script works correctly
    warnings := []          ; Array of warning strings from server or validation
    sessionId := ""         ; Session ID for follow-up refinement requests
    workflowId := ""        ; Workflow ID this script was generated from
    executionResult := ""   ; Populated after ExecuteScript() -- ExecutionResult or ""
    executed := 0           ; Boolean: whether ExecuteScript() has been called
    metadata := Map()       ; Raw metadata from server response

    __New() {
        this.warnings := []
        this.metadata := Map()
    }
}


; ============================================================================
; AIEngine -- Main AI script generation and execution engine
; ============================================================================
class AIEngine {

    ; --- Dependencies ---
    httpClient := ""            ; HttpClient instance for server communication
    config := Map()             ; Configuration map

    ; --- Configuration (with defaults) ---
    executionTimeout := 60      ; Kill script process after this many seconds
    maxRetries := 4             ; Max error recovery attempts before giving up
    autoRefine := true          ; Automatically refine on execution failure
    scriptSaveDir := ""         ; Directory for saving generated scripts
    ahkExePath := ""            ; Path to AutoHotkey64.exe interpreter
    serverBasePath := "/api/v1" ; API base path on the server

    ; --- State ---
    lastResult := ""            ; Most recent ScriptResult
    lastSessionId := ""         ; Most recent session ID
    isProcessing := 0           ; 1 = waiting for server response
    _runningPid := 0            ; PID of currently executing script (0 = none)

    ; --- Safety: Blocked patterns (severity "block" = never execute) ---
    static BLOCKED_PATTERNS := [
        Map("pattern", "i)#Requires\s+Admin",
            "severity", "block",
            "reason", "Script requests administrator privileges"),
        Map("pattern", "i)RunAs\b",
            "severity", "block",
            "reason", "Script attempts privilege escalation via RunAs"),
        Map("pattern", "i)FileDelete\s*\(?[`"']?[A-Z]:\\(Windows|Program Files)",
            "severity", "block",
            "reason", "Script deletes system files"),
        Map("pattern", "i)DirDelete\s*\(?[`"']?[A-Z]:\\(Windows|Program Files)",
            "severity", "block",
            "reason", "Script deletes system directories"),
        Map("pattern", "i)RegDelete\b",
            "severity", "block",
            "reason", "Script deletes Windows registry keys"),
        Map("pattern", "i)Download\b.*\.(exe|bat|cmd|ps1|vbs|scr|msi|dll)",
            "severity", "block",
            "reason", "Script downloads executable files from the internet"),
        Map("pattern", "i)Run.*cmd\.exe.*/[cC]\s.*(del\s|rd\s|rmdir\s|format\s)",
            "severity", "block",
            "reason", "Script runs destructive shell commands via cmd.exe"),
        Map("pattern", "i)Run.*powershell.*Remove-Item",
            "severity", "block",
            "reason", "Script runs destructive PowerShell commands"),
        Map("pattern", "i)Run.*(curl|wget|Invoke-WebRequest).*\|\s*(cmd|powershell|sh)",
            "severity", "block",
            "reason", "Script downloads and pipes to shell execution"),
        Map("pattern", "i)SendMessage\s.*0x0112",
            "severity", "block",
            "reason", "Script sends WM_SYSCOMMAND to system windows")
    ]

    ; --- Safety: Warning patterns (severity "warn" = ask user before executing) ---
    static WARNING_PATTERNS := [
        Map("pattern", "i)RegWrite\b",
            "severity", "warn",
            "reason", "Script modifies the Windows registry"),
        Map("pattern", "i)FileDelete\b",
            "severity", "warn",
            "reason", "Script deletes files"),
        Map("pattern", "i)DirDelete\b",
            "severity", "warn",
            "reason", "Script deletes directories"),
        Map("pattern", "i)DllCall\b(?!.*\b(User32|Kernel32|Gdi32|Shell32|Advapi32|Comctl32)\b)",
            "severity", "warn",
            "reason", "Script calls DllCall to a non-standard DLL"),
        Map("pattern", "i)Run\b.*\b(powershell|cmd|wscript|cscript)\b",
            "severity", "warn",
            "reason", "Script launches a shell or scripting interpreter"),
        Map("pattern", "i)#Include\b(?!.*\blib\\|.*\bvendor\\)",
            "severity", "warn",
            "reason", "Script includes external files outside the project"),
        Map("pattern", "i)Loop\s*\{?\s*$",
            "severity", "warn",
            "reason", "Script contains an unconditional infinite loop"),
        Map("pattern", "i)Download\b",
            "severity", "warn",
            "reason", "Script downloads files from the internet"),
        Map("pattern", "i)Run\b",
            "severity", "warn",
            "reason", "Script launches external processes")
    ]

    ; ====================================================================
    ; Constructor
    ; ====================================================================

    /**
     * Initialize the AI Engine.
     *
     * @param httpClient  HttpClient instance (already configured with auth)
     * @param config      Map of configuration options:
     *   "executionTimeout"  (Integer) Seconds before killing a running script (default: 60)
     *   "maxRetries"        (Integer) Max refinement attempts on failure (default: 4)
     *   "autoRefine"        (Integer) Auto-refine on execution failure (default: 1)
     *   "scriptSaveDir"     (String)  Directory for generated scripts
     *   "ahkExePath"        (String)  Path to AutoHotkey64.exe
     *   "serverBasePath"    (String)  API base path (default: "/api/v1")
     */
    __New(httpClient, config := Map()) {
        if (!httpClient)
            throw AIEngineError("HttpClient instance is required", "validation")

        this.httpClient := httpClient
        this.config := config

        ; Apply configuration with defaults
        if (config is Map) {
            if config.Has("executionTimeout")
                this.executionTimeout := Integer(config["executionTimeout"])
            if config.Has("maxRetries")
                this.maxRetries := Integer(config["maxRetries"])
            if config.Has("autoRefine")
                this.autoRefine := !!(config["autoRefine"])
            if config.Has("scriptSaveDir")
                this.scriptSaveDir := String(config["scriptSaveDir"])
            if config.Has("ahkExePath")
                this.ahkExePath := String(config["ahkExePath"])
            if config.Has("serverBasePath")
                this.serverBasePath := String(config["serverBasePath"])
        }

        ; Default scriptSaveDir to scripts/ in the app directory
        if (this.scriptSaveDir = "")
            this.scriptSaveDir := A_ScriptDir "\scripts"

        ; Default ahkExePath to bundled AutoHotkey64.exe
        if (this.ahkExePath = "")
            this.ahkExePath := A_ScriptDir "\AutoHotkey64.exe"

        ; Ensure scripts directory exists
        if !DirExist(this.scriptSaveDir) {
            try DirCreate(this.scriptSaveDir)
        }
    }

    ; ====================================================================
    ; GenerateScript -- Main entry point: workflow -> generated AHK script
    ; ====================================================================

    /**
     * Generate an AHK v2 script from a serialized workflow.
     *
     * Sends the workflow to the Hercules proxy server which forwards it
     * to Claude for script generation. Returns a ScriptResult with the
     * generated code, explanation, and confidence score.
     *
     * @param workflow    Map -- Serialized workflow from WorkflowSerializer.SerializeWorkflow()
     *                    Must contain "workflow" key with actions array, or be the
     *                    workflow object itself.
     * @param sessionId   (Optional) Session ID for conversational context.
     *                    If provided, the server uses the session's history.
     * @param options     (Optional) Map of generation options:
     *                    "style"              : "minimal"|"robust"|"production" (default: "robust")
     *                    "error_handling"     : "none"|"basic"|"standard"|"comprehensive"
     *                    "adaptiveness"       : "low"|"medium"|"high"
     *                    "include_comments"   : 1|0 (default: 1)
     *                    "timing_mode"        : "fast"|"natural"|"cautious"
     *                    "target_ahk_version" : "2.0.21" (default)
     * @param instructions (Optional) Additional natural-language instructions for Claude.
     * @returns ScriptResult
     * @throws AIEngineError on failure
     */
    GenerateScript(workflow, sessionId := "", options := "", instructions := "") {
        if (this.isProcessing)
            throw AIEngineError("A generation request is already in progress", "validation")

        this.isProcessing := 1

        try {
            ; Build the request payload
            payload := Map()

            ; Determine if workflow is wrapped or bare
            if (workflow is Map) && workflow.Has("workflow") {
                payload["workflow"] := workflow["workflow"]
                if workflow.Has("tags")
                    payload["tags"] := workflow["tags"]
            } else if (workflow is Map) {
                payload["workflow"] := workflow
            } else {
                throw AIEngineError("Workflow must be a Map object", "validation")
            }

            ; Add session ID if provided
            if (sessionId != "") {
                payload["session_id"] := sessionId
            }

            ; Add generation options (or use defaults)
            if (options is Map) && (options.Count > 0) {
                payload["options"] := options
            } else {
                payload["options"] := Map(
                    "style", "robust",
                    "error_handling", "standard",
                    "adaptiveness", "medium",
                    "include_comments", 1,
                    "timing_mode", "natural",
                    "target_ahk_version", "2.0.21"
                )
            }

            ; Add natural-language instructions
            if (instructions != "")
                payload["instructions"] := instructions

            ; Send to server
            response := this.httpClient.Post(this.serverBasePath . "/generate", payload)

            ; Parse the response into a ScriptResult
            result := this._ParseGenerateResponse(response)

            ; Run client-side safety validation
            validationResult := this.ValidateGeneratedScript(result.scriptCode)
            if (validationResult.Has("blocked") && validationResult["blocked"].Length > 0) {
                for _, reason in validationResult["blocked"] {
                    result.warnings.Push("[BLOCKED] " . reason)
                }
            }
            if (validationResult.Has("warnings") && validationResult["warnings"].Length > 0) {
                for _, reason in validationResult["warnings"] {
                    result.warnings.Push("[WARNING] " . reason)
                }
            }

            this.lastResult := result
            if (result.sessionId != "")
                this.lastSessionId := result.sessionId

            return result

        } catch HttpError as httpErr {
            throw AIEngineError(
                "Script generation failed: " . httpErr.Message,
                "network",
                httpErr.isRetryable
            )
        } finally {
            this.isProcessing := 0
        }
    }

    ; ====================================================================
    ; ExecuteScript -- Execute a generated AHK script safely
    ; ====================================================================

    /**
     * Execute an AHK v2 script in a sandboxed subprocess.
     *
     * Safety protocol:
     *   1. Validate the script for dangerous patterns
     *   2. If blocked patterns found, refuse execution
     *   3. Write script to a temp file
     *   4. Launch via AutoHotkey64.exe subprocess
     *   5. Capture stdout/stderr via temp files
     *   6. Wait for completion or timeout
     *   7. Clean up temp files
     *
     * @param scriptCode   The AHK v2 script text to execute.
     * @param sandboxed    (Boolean, default true) If true, validates before execution.
     *                     Set to false ONLY for scripts the user has explicitly reviewed.
     * @returns ExecutionResult
     * @throws AIEngineError if script is blocked by safety validation
     */
    ExecuteScript(scriptCode, sandboxed := true) {
        execResult := ExecutionResult()

        if (scriptCode = "")
            throw AIEngineError("Cannot execute an empty script", "validation")

        ; Step 1: Safety validation
        if (sandboxed) {
            validation := this.ValidateGeneratedScript(scriptCode)
            if (validation["blocked"].Length > 0) {
                reasons := ""
                for _, reason in validation["blocked"] {
                    reasons .= "- " . reason . "`n"
                }
                throw AIEngineError(
                    "Script blocked by safety validation:`n" . reasons,
                    "safety"
                )
            }
        }

        ; Step 2: Generate unique temp file paths
        timestamp := A_TickCount
        scriptPath := A_Temp "\claude-ahk-run-" . timestamp . ".ahk"
        stdoutPath := A_Temp "\claude-ahk-stdout-" . timestamp . ".txt"
        stderrPath := A_Temp "\claude-ahk-stderr-" . timestamp . ".txt"
        execResult.scriptPath := scriptPath

        try {
            ; Step 3: Write the script to temp file
            ; Wrap the user script in a launcher that captures output
            launcherCode := this._BuildLauncherScript(scriptCode, stdoutPath, stderrPath)
            scriptFile := FileOpen(scriptPath, "w", "UTF-8")
            scriptFile.Write(launcherCode)
            scriptFile.Close()

            ; Step 4: Launch the script subprocess
            startTime := A_TickCount

            ; Build the command line
            ; Use cmd.exe /c to launch and capture exit code
            cmdLine := '"' . this.ahkExePath . '" /script "' . scriptPath . '"'

            try {
                Run(cmdLine, A_Temp, "Hide", &pid)
                execResult.pid := pid
                this._runningPid := pid
            } catch Error as runErr {
                throw AIEngineError(
                    "Failed to launch script: " . runErr.Message,
                    "execution"
                )
            }

            ; Step 5: Wait for completion or timeout
            timeoutMs := this.executionTimeout * 1000
            waitResult := ProcessWaitClose(pid, this.executionTimeout)

            endTime := A_TickCount
            execResult.durationMs := endTime - startTime

            if (waitResult = 0) {
                ; Process did not exit within timeout -- kill it
                execResult.timedOut := 1
                execResult.success := 0
                try ProcessClose(pid)
                ; Give it a moment to die
                Sleep(500)
                try ProcessClose(pid)  ; Force if still alive
            } else {
                execResult.timedOut := 0
            }

            ; Step 6: Read captured output
            try {
                if FileExist(stdoutPath) {
                    execResult.stdout := Trim(FileRead(stdoutPath, "UTF-8"))
                }
            }
            try {
                if FileExist(stderrPath) {
                    execResult.stderr := Trim(FileRead(stderrPath, "UTF-8"))
                }
            }

            ; Determine exit code from stderr content (AHK errors go to stderr)
            if (!execResult.timedOut) {
                if (execResult.stderr = "" || !InStr(execResult.stderr, "Error")) {
                    execResult.exitCode := 0
                    execResult.success := 1
                } else {
                    execResult.exitCode := 1
                    execResult.success := 0
                }
            }

            this._runningPid := 0

        } finally {
            ; Step 7: Clean up temp files
            this._CleanupTempFile(scriptPath)
            this._CleanupTempFile(stdoutPath)
            this._CleanupTempFile(stderrPath)
        }

        ; Update lastResult if we have one
        if (this.lastResult is ScriptResult) {
            this.lastResult.executionResult := execResult
            this.lastResult.executed := 1
        }

        return execResult
    }

    ; ====================================================================
    ; RefineScript -- Send refinement request to improve a script
    ; ====================================================================

    /**
     * Refine a previously generated script within a session.
     *
     * Sends feedback (error output, user description, or enhancement request)
     * to the server. The server maintains conversation history so Claude can
     * iterate on its previous output.
     *
     * @param scriptCode   The current script code being refined.
     * @param feedback     String -- Natural language description of the issue or desired change.
     * @param sessionId    Session ID from the original GenerateScript call.
     * @param errorContext  (Optional) Map with additional error details:
     *                     "error_output"           : AHK error text
     *                     "screenshot_description" : What the screen looked like
     *                     "partial_success"        : What worked before failure
     *                     "environment_changes"    : What changed since last generation
     * @returns ScriptResult -- Updated script with fixes applied
     * @throws AIEngineError on failure
     */
    RefineScript(scriptCode, feedback, sessionId, errorContext := "") {
        if (sessionId = "")
            throw AIEngineError("Session ID is required for refinement", "validation")

        if (feedback = "")
            throw AIEngineError("Feedback message is required for refinement", "validation")

        this.isProcessing := 1

        try {
            ; Build the refinement payload per PROTOCOL.md section 5.3
            payload := Map(
                "message", feedback
            )

            ; Add generation options
            payload["options"] := Map(
                "style", "robust",
                "error_handling", "comprehensive",
                "adaptiveness", "high",
                "include_comments", 1,
                "timing_mode", "natural",
                "target_ahk_version", "2.0.21"
            )

            ; Add structured error context if provided
            if (errorContext is Map) && (errorContext.Count > 0) {
                payload["additional_context"] := errorContext
            }

            ; POST to /api/v1/session/{sessionId}/refine
            path := this.serverBasePath . "/session/" . sessionId . "/refine"
            response := this.httpClient.Post(path, payload)

            ; Parse the refinement response (same shape as generate)
            result := this._ParseGenerateResponse(response)
            result.sessionId := sessionId

            ; Validate the refined script
            validation := this.ValidateGeneratedScript(result.scriptCode)
            if (validation["blocked"].Length > 0) {
                for _, reason in validation["blocked"] {
                    result.warnings.Push("[BLOCKED] " . reason)
                }
            }
            if (validation["warnings"].Length > 0) {
                for _, reason in validation["warnings"] {
                    result.warnings.Push("[WARNING] " . reason)
                }
            }

            this.lastResult := result
            return result

        } catch HttpError as httpErr {
            throw AIEngineError(
                "Script refinement failed: " . httpErr.Message,
                "network",
                httpErr.isRetryable
            )
        } finally {
            this.isProcessing := 0
        }
    }

    ; ====================================================================
    ; AnalyzeWorkflow -- Request workflow analysis (no code generation)
    ; ====================================================================

    /**
     * Request a structured analysis of a workflow from Claude.
     *
     * This corresponds to Stage 1 of the prompt chaining strategy.
     * The server forces Claude to use the analyze_workflow tool,
     * producing a structured breakdown of user intent, patterns,
     * risk factors, and targeting strategy.
     *
     * @param workflow  Map -- Serialized workflow (same format as GenerateScript)
     * @returns Map with analysis fields:
     *   "summary"          : String  -- One-paragraph description
     *   "intent"           : String  -- Workflow category
     *   "steps"            : Array   -- Logical step breakdown
     *   "patterns"         : Map     -- Detected patterns
     *   "risk_factors"     : Array   -- Potential failure points
     *   "target_strategy"  : String  -- Recommended script approach
     * @throws AIEngineError on failure
     */
    AnalyzeWorkflow(workflow) {
        this.isProcessing := 1

        try {
            ; Build the analysis payload
            payload := Map()

            if (workflow is Map) && workflow.Has("workflow") {
                payload["workflow"] := workflow["workflow"]
            } else if (workflow is Map) {
                payload["workflow"] := workflow
            } else {
                throw AIEngineError("Workflow must be a Map object", "validation")
            }

            payload["analysis_only"] := 1

            ; POST to the generate endpoint with analysis_only flag
            ; The server will use Stage 1 prompt chaining (analyze only)
            response := this.httpClient.Post(this.serverBasePath . "/generate", payload)

            ; Parse the analysis response
            jsonBody := ""
            if (response is Map) && response.Has("json") && (response["json"] is Map) {
                jsonBody := response["json"]
            } else {
                throw AIEngineError("Invalid analysis response from server", "parse")
            }

            ; Extract the analysis fields
            analysis := Map()
            for _, key in ["summary", "intent", "steps", "patterns", "risk_factors", "target_strategy"] {
                if jsonBody.Has(key)
                    analysis[key] := jsonBody[key]
                else
                    analysis[key] := ""
            }

            ; Also capture workflow_id and session_id if present
            if jsonBody.Has("workflow_id")
                analysis["workflow_id"] := jsonBody["workflow_id"]
            if jsonBody.Has("session_id") {
                analysis["session_id"] := jsonBody["session_id"]
                this.lastSessionId := jsonBody["session_id"]
            }

            return analysis

        } catch HttpError as httpErr {
            throw AIEngineError(
                "Workflow analysis failed: " . httpErr.Message,
                "network",
                httpErr.isRetryable
            )
        } finally {
            this.isProcessing := 0
        }
    }

    ; ====================================================================
    ; ValidateGeneratedScript -- Safety check on AHK v2 script text
    ; ====================================================================

    /**
     * Validate a generated AHK v2 script for safety and basic syntax.
     *
     * Checks the script text against blocked and warning pattern lists.
     * Also validates basic structural requirements.
     *
     * @param scriptCode  String -- The AHK v2 script to validate.
     * @returns Map with keys:
     *   "safe"      : Integer (1 if no blocked patterns found, 0 if blocked)
     *   "blocked"   : Array of reason strings for blocked patterns
     *   "warnings"  : Array of reason strings for warning patterns
     *   "info"      : Array of informational notes (size, line count, etc.)
     */
    ValidateGeneratedScript(scriptCode) {
        result := Map(
            "safe", 1,
            "blocked", [],
            "warnings", [],
            "info", []
        )

        if (scriptCode = "") {
            result["blocked"].Push("Script is empty")
            result["safe"] := 0
            return result
        }

        ; Check script size (max 100KB)
        scriptSize := StrLen(scriptCode) * 2  ; Approximate byte size (UTF-16)
        if (scriptSize > 102400) {
            result["blocked"].Push("Script exceeds 100KB size limit (" . Round(scriptSize / 1024, 1) . "KB)")
            result["safe"] := 0
        }

        ; Count lines for info
        lineCount := 1
        loop parse scriptCode, "`n" {
            lineCount := A_Index
        }
        result["info"].Push("Script is " . lineCount . " lines, ~" . Round(scriptSize / 1024, 1) . "KB")

        ; Check blocked patterns -- these prevent execution entirely
        for _, rule in AIEngine.BLOCKED_PATTERNS {
            if RegExMatch(scriptCode, rule["pattern"]) {
                result["blocked"].Push(rule["reason"])
                result["safe"] := 0
            }
        }

        ; Check warning patterns -- user should review before execution
        for _, rule in AIEngine.WARNING_PATTERNS {
            if RegExMatch(scriptCode, rule["pattern"]) {
                result["warnings"].Push(rule["reason"])
            }
        }

        ; Check for #Requires AutoHotkey v2.0 header (informational)
        if !RegExMatch(scriptCode, "i)#Requires\s+AutoHotkey\s+v2") {
            result["info"].Push("Script does not include #Requires AutoHotkey v2.0 directive")
        }

        ; Check for infinite loops without exit conditions
        ; Look for "Loop {" or "while true {" without any break/return/ExitApp nearby
        if RegExMatch(scriptCode, "im)^\s*Loop\s*\{") {
            ; Check if there is at least one break, return, or ExitApp in the script
            if !RegExMatch(scriptCode, "i)\b(break|return|ExitApp)\b") {
                result["warnings"].Push("Script contains a Loop block but no break/return/ExitApp was found")
            }
        }
        if RegExMatch(scriptCode, "im)^\s*while\s+(true|1)\s*\{") {
            if !RegExMatch(scriptCode, "i)\b(break|return|ExitApp)\b") {
                result["warnings"].Push("Script contains while(true) but no break/return/ExitApp was found")
            }
        }

        return result
    }

    ; ====================================================================
    ; GetSessionHistory -- Retrieve conversation history for a session
    ; ====================================================================

    /**
     * Get the message history for a session.
     *
     * @param sessionId  The session ID to retrieve.
     * @returns Map containing session data:
     *   "session_id"    : String
     *   "name"          : String
     *   "messages"      : Array of message Maps
     *   "message_count" : Integer
     *   "status"        : String ("active" | "archived")
     * @throws AIEngineError on failure
     */
    GetSessionHistory(sessionId) {
        if (sessionId = "")
            throw AIEngineError("Session ID is required", "validation")

        try {
            path := this.serverBasePath . "/session/" . sessionId
            response := this.httpClient.Get(path)

            if (response is Map) && response.Has("json") && (response["json"] is Map) {
                return response["json"]
            }

            throw AIEngineError("Invalid session response from server", "parse")

        } catch HttpError as httpErr {
            throw AIEngineError(
                "Failed to retrieve session history: " . httpErr.Message,
                "network",
                httpErr.isRetryable
            )
        }
    }

    ; ====================================================================
    ; StopExecution -- Kill a running script process
    ; ====================================================================

    /**
     * Terminate the currently executing script subprocess, if any.
     *
     * @returns Integer -- 1 if a process was killed, 0 if nothing was running.
     */
    StopExecution() {
        if (this._runningPid = 0)
            return 0

        try {
            ProcessClose(this._runningPid)
            Sleep(200)
            ; Force kill if still alive
            try ProcessClose(this._runningPid)
        }

        this._runningPid := 0
        return 1
    }

    ; ====================================================================
    ; AutoExecuteWithRefinement -- Execute and auto-fix on failure
    ; ====================================================================

    /**
     * Execute a script and, if it fails, automatically send the error
     * back to Claude for iterative refinement. Implements the error
     * recovery protocol from PROMPT_ENGINEERING.md section 6.
     *
     * @param scriptCode  The AHK v2 script to execute.
     * @param sessionId   Session ID for refinement context.
     * @param maxAttempts  (Optional) Max attempts (default: this.maxRetries)
     * @returns Map with keys:
     *   "success"       : Integer (1 if script succeeded, 0 if all attempts failed)
     *   "attempts"      : Integer (how many attempts were made)
     *   "finalResult"   : ScriptResult (the last result, successful or not)
     *   "execResult"    : ExecutionResult (the last execution result)
     *   "history"       : Array of Maps with attempt details
     */
    AutoExecuteWithRefinement(scriptCode, sessionId := "", maxAttempts := 0) {
        if (maxAttempts <= 0)
            maxAttempts := this.maxRetries

        if (sessionId = "")
            sessionId := this.lastSessionId

        currentScript := scriptCode
        history := []
        lastExecResult := ""
        lastScriptResult := ""

        loop maxAttempts {
            attemptNum := A_Index

            ; Execute the current script
            try {
                execResult := this.ExecuteScript(currentScript, true)
                lastExecResult := execResult
            } catch AIEngineError as safetyErr {
                ; Script was blocked by safety validation -- cannot retry
                history.Push(Map(
                    "attempt", attemptNum,
                    "action", "blocked",
                    "error", safetyErr.Message
                ))
                return Map(
                    "success", 0,
                    "attempts", attemptNum,
                    "finalResult", this.lastResult,
                    "execResult", "",
                    "history", history
                )
            }

            ; Check if execution succeeded
            if (execResult.success) {
                history.Push(Map(
                    "attempt", attemptNum,
                    "action", "success",
                    "durationMs", execResult.durationMs
                ))
                return Map(
                    "success", 1,
                    "attempts", attemptNum,
                    "finalResult", this.lastResult,
                    "execResult", execResult,
                    "history", history
                )
            }

            ; Execution failed -- log the attempt
            history.Push(Map(
                "attempt", attemptNum,
                "action", "failed",
                "error", execResult.stderr,
                "timedOut", execResult.timedOut,
                "durationMs", execResult.durationMs
            ))

            ; If this was the last attempt, do not try to refine
            if (attemptNum >= maxAttempts)
                break

            ; If no session, cannot refine
            if (sessionId = "")
                break

            ; Build error context for refinement
            errorContext := this._BuildErrorContext(execResult, attemptNum)

            ; Build escalating feedback messages per the iterative fix protocol
            feedback := this._BuildRefinementFeedback(execResult, attemptNum, maxAttempts)

            ; Send refinement request
            try {
                refinedResult := this.RefineScript(currentScript, feedback, sessionId, errorContext)
                lastScriptResult := refinedResult

                if (refinedResult.scriptCode != "") {
                    currentScript := refinedResult.scriptCode
                    history.Push(Map(
                        "attempt", attemptNum,
                        "action", "refined",
                        "confidence", refinedResult.confidenceScore
                    ))
                } else {
                    ; Refinement returned no script -- give up
                    break
                }
            } catch AIEngineError {
                ; Refinement request failed -- give up
                break
            }
        }

        ; All attempts exhausted
        return Map(
            "success", 0,
            "attempts", history.Length,
            "finalResult", lastScriptResult != "" ? lastScriptResult : this.lastResult,
            "execResult", lastExecResult,
            "history", history
        )
    }

    ; ====================================================================
    ; CreateSession -- Create a new conversation session
    ; ====================================================================

    /**
     * Create a new session for iterative script refinement.
     *
     * @param workflowId     (Optional) Workflow ID to associate with the session.
     * @param name           (Optional) Human-friendly session name.
     * @param systemContext   (Optional) Additional context about the user's environment.
     * @returns Map with session data including "session_id".
     * @throws AIEngineError on failure
     */
    CreateSession(workflowId := "", name := "", systemContext := "") {
        payload := Map()

        if (workflowId != "")
            payload["workflow_id"] := workflowId
        if (name != "")
            payload["name"] := name
        if (systemContext != "")
            payload["system_context"] := systemContext

        try {
            response := this.httpClient.Post(this.serverBasePath . "/session", payload)

            if (response is Map) && response.Has("json") && (response["json"] is Map) {
                sessionData := response["json"]
                if sessionData.Has("session_id")
                    this.lastSessionId := sessionData["session_id"]
                return sessionData
            }

            throw AIEngineError("Invalid session creation response", "parse")

        } catch HttpError as httpErr {
            throw AIEngineError(
                "Failed to create session: " . httpErr.Message,
                "network",
                httpErr.isRetryable
            )
        }
    }

    ; ====================================================================
    ; SaveScript -- Save a generated script to disk
    ; ====================================================================

    /**
     * Save a generated script and its metadata to the scripts directory.
     *
     * Creates two files:
     *   scripts/{name}.ahk       -- The AHK v2 script
     *   scripts/{name}.meta.json -- Metadata (explanation, confidence, etc.)
     *
     * @param scriptCode   The script text to save.
     * @param name         Filename (without extension). Defaults to timestamp-based name.
     * @param metadata     (Optional) Map of metadata to save alongside.
     * @returns Map("scriptPath", path, "metaPath", path)
     */
    SaveScript(scriptCode, name := "", metadata := "") {
        if (scriptCode = "")
            throw AIEngineError("Cannot save an empty script", "validation")

        ; Generate a default name if not provided
        if (name = "")
            name := "generated-" . FormatTime(, "yyyyMMdd-HHmmss")

        ; Sanitize filename (remove unsafe characters)
        name := RegExReplace(name, '[<>:"/\\|?*]', "_")

        ; Build paths
        scriptPath := this.scriptSaveDir . "\" . name . ".ahk"
        metaPath := this.scriptSaveDir . "\" . name . ".meta.json"

        ; Write the script file
        try {
            scriptFile := FileOpen(scriptPath, "w", "UTF-8")
            scriptFile.Write(scriptCode)
            scriptFile.Close()
        } catch Error as writeErr {
            throw AIEngineError("Failed to save script: " . writeErr.Message, "execution")
        }

        ; Write metadata if available
        if (metadata is Map) && (metadata.Count > 0) {
            try {
                metaJson := Jxon_Dump(metadata, 2)
                metaFile := FileOpen(metaPath, "w", "UTF-8")
                metaFile.Write(metaJson)
                metaFile.Close()
            }
        } else if (this.lastResult is ScriptResult) {
            ; Build metadata from the last result
            meta := Map(
                "scriptId", this.lastResult.scriptId,
                "explanation", this.lastResult.explanation,
                "confidenceScore", this.lastResult.confidenceScore,
                "warnings", this.lastResult.warnings,
                "sessionId", this.lastResult.sessionId,
                "workflowId", this.lastResult.workflowId,
                "savedAt", FormatTime(, "yyyy-MM-dd HH:mm:ss")
            )
            try {
                metaJson := Jxon_Dump(meta, 2)
                metaFile := FileOpen(metaPath, "w", "UTF-8")
                metaFile.Write(metaJson)
                metaFile.Close()
            }
        }

        return Map("scriptPath", scriptPath, "metaPath", metaPath)
    }

    ; ====================================================================
    ; GetLastResult -- Access the most recent generation result
    ; ====================================================================

    /**
     * @returns ScriptResult or "" if no generation has occurred.
     */
    GetLastResult() {
        return this.lastResult
    }

    ; ====================================================================
    ; IsProcessing -- Check if a request is in flight
    ; ====================================================================

    /**
     * @returns Integer -- 1 if a request is in progress, 0 otherwise.
     */
    IsProcessing() {
        return this.isProcessing
    }

    ; ====================================================================
    ; INTERNAL METHODS
    ; ====================================================================

    ; --------------------------------------------------------------------
    ; _ParseGenerateResponse -- Parse server response into ScriptResult
    ; --------------------------------------------------------------------

    /**
     * Parse the HTTP response from /generate or /session/{id}/refine
     * into a structured ScriptResult object.
     *
     * Expected response JSON keys (per PROTOCOL.md section 4.1):
     *   generation_id, script, explanation, confidence, metadata,
     *   workflow_id, session_id, warnings, changes_summary
     *
     * @param response  Map -- HttpResponse from HttpClient (with "json" key)
     * @returns ScriptResult
     * @throws AIEngineError if response cannot be parsed
     */
    _ParseGenerateResponse(response) {
        result := ScriptResult()

        ; Extract the JSON body
        jsonBody := ""
        if (response is Map) && response.Has("json") {
            jsonBody := response["json"]
        }

        if !(jsonBody is Map) {
            throw AIEngineError(
                "Server response is not valid JSON or missing expected structure",
                "parse"
            )
        }

        ; Map the server response fields to ScriptResult properties
        ; The server uses "generation_id" or "script_id" for the ID
        if jsonBody.Has("generation_id")
            result.scriptId := String(jsonBody["generation_id"])
        else if jsonBody.Has("script_id")
            result.scriptId := String(jsonBody["script_id"])

        ; Script code -- the server uses "script" or "script_code"
        if jsonBody.Has("script")
            result.scriptCode := String(jsonBody["script"])
        else if jsonBody.Has("script_code")
            result.scriptCode := String(jsonBody["script_code"])

        ; Explanation
        if jsonBody.Has("explanation")
            result.explanation := String(jsonBody["explanation"])

        ; Confidence score -- "confidence" or "confidence_score"
        if jsonBody.Has("confidence")
            result.confidenceScore := this._ToFloat(jsonBody["confidence"])
        else if jsonBody.Has("confidence_score")
            result.confidenceScore := this._ToFloat(jsonBody["confidence_score"])

        ; Session ID
        if jsonBody.Has("session_id")
            result.sessionId := String(jsonBody["session_id"])

        ; Workflow ID
        if jsonBody.Has("workflow_id")
            result.workflowId := String(jsonBody["workflow_id"])

        ; Warnings (Array)
        if jsonBody.Has("warnings") && (jsonBody["warnings"] is Array) {
            for _, warning in jsonBody["warnings"] {
                result.warnings.Push(String(warning))
            }
        }

        ; Metadata (Map)
        if jsonBody.Has("metadata") && (jsonBody["metadata"] is Map) {
            result.metadata := jsonBody["metadata"]
        }

        return result
    }

    ; --------------------------------------------------------------------
    ; _BuildLauncherScript -- Wrap user script with output capture
    ; --------------------------------------------------------------------

    /**
     * Build a launcher script that wraps the user script with:
     *   - Output redirection (stdout/stderr to temp files)
     *   - Error capture via OnError callback
     *   - The user's actual script code
     *
     * @param scriptCode   The user's AHK v2 script
     * @param stdoutPath   Path for captured stdout
     * @param stderrPath   Path for captured stderr
     * @returns String -- Complete launcher script
     */
    _BuildLauncherScript(scriptCode, stdoutPath, stderrPath) {
        ; AHK v2 does NOT use backslash as escape char, so no escaping needed.
        ; Paths are embedded directly in strings.

        ; Build the launcher that wraps the user script
        ; The launcher installs a global error handler to capture errors
        ; to the stderr file, then runs the user script.
        launcher := "#Requires AutoHotkey v2.0`n"
        launcher .= "#SingleInstance Force`n"
        launcher .= "`n"
        launcher .= "; --- Claude AHK Launcher: Error Capture ---`n"
        launcher .= "_claude_ahk_stderr_path := " Chr(34) stderrPath Chr(34) "`n"
        launcher .= "_claude_ahk_stdout_path := " Chr(34) stdoutPath Chr(34) "`n"
        launcher .= "`n"
        launcher .= "OnError(_ClaudeAHK_ErrorHandler)`n"
        launcher .= "`n"
        launcher .= "_ClaudeAHK_ErrorHandler(exc, mode) {`n"
        launcher .= "    global _claude_ahk_stderr_path`n"
        launcher .= "    errMsg := " Chr(34) "Error: " Chr(34) " . exc.Message`n"
        launcher .= "    errMsg .= " Chr(34) "``nFile: " Chr(34) " . exc.File`n"
        launcher .= "    errMsg .= " Chr(34) "``nLine: " Chr(34) " . exc.Line`n"
        launcher .= "    errMsg .= " Chr(34) "``nWhat: " Chr(34) " . exc.What`n"
        launcher .= "    try {`n"
        launcher .= "        f := FileOpen(_claude_ahk_stderr_path, " Chr(34) "w" Chr(34) ", " Chr(34) "UTF-8" Chr(34) ")`n"
        launcher .= "        f.Write(errMsg)`n"
        launcher .= "        f.Close()`n"
        launcher .= "    }`n"
        launcher .= "    return 1`n"
        launcher .= "}`n"
        launcher .= "`n"
        launcher .= "; --- User Script Begins ---`n"
        launcher .= "`n"

        ; Strip any #Requires and #SingleInstance from the user script
        ; to avoid conflicts with the launcher directives
        cleanedScript := scriptCode
        cleanedScript := RegExReplace(cleanedScript, "im)^\s*#Requires\s+AutoHotkey\s+v2[^\n]*\n?", "")
        cleanedScript := RegExReplace(cleanedScript, "im)^\s*#SingleInstance\s+[^\n]*\n?", "")

        launcher .= cleanedScript
        launcher .= "`n"
        launcher .= "`n; --- Claude AHK Launcher: Write success marker ---`n"
        launcher .= "try {`n"
        launcher .= "    f := FileOpen(_claude_ahk_stdout_path, " Chr(34) "w" Chr(34) ", " Chr(34) "UTF-8" Chr(34) ")`n"
        launcher .= "    f.Write(" Chr(34) "Script completed successfully" Chr(34) ")`n"
        launcher .= "    f.Close()`n"
        launcher .= "}`n"

        return launcher
    }

    ; --------------------------------------------------------------------
    ; _BuildErrorContext -- Build structured error context for refinement
    ; --------------------------------------------------------------------

    /**
     * Build the additional_context Map for a refinement request
     * based on execution failure details.
     *
     * @param execResult   ExecutionResult from the failed execution
     * @param attemptNum   Which attempt this is (1-based)
     * @returns Map -- Error context for refinement payload
     */
    _BuildErrorContext(execResult, attemptNum) {
        context := Map()

        ; Include error output
        if (execResult.stderr != "")
            context["error_output"] := execResult.stderr

        ; Include stdout if it has useful info
        if (execResult.stdout != "")
            context["partial_success"] := execResult.stdout

        ; Note if execution timed out
        if (execResult.timedOut)
            context["environment_changes"] := "Script timed out after " . this.executionTimeout . " seconds"

        return context
    }

    ; --------------------------------------------------------------------
    ; _BuildRefinementFeedback -- Build escalating feedback messages
    ; --------------------------------------------------------------------

    /**
     * Build the feedback message for refinement, escalating with each attempt
     * per the iterative fix protocol in PROMPT_ENGINEERING.md section 6.
     *
     * Attempt 1: Standard error report
     * Attempt 2: Detailed diagnostics, request fundamentally different approach
     * Attempt 3: Request decomposition to isolate failing step
     * Attempt 4+: Human handoff (should not reach here, but just in case)
     *
     * @param execResult   ExecutionResult from the failed execution
     * @param attemptNum   Which attempt this is (1-based)
     * @param maxAttempts  Total max attempts
     * @returns String -- Feedback message for the refinement request
     */
    _BuildRefinementFeedback(execResult, attemptNum, maxAttempts) {
        feedback := ""

        if (attemptNum = 1) {
            ; Attempt 1: Standard error report
            feedback := "The script encountered an error during execution.`n`n"

            if (execResult.timedOut) {
                feedback .= "Error: Script timed out after " . this.executionTimeout . " seconds.`n"
                feedback .= "The script did not complete within the expected time. "
                feedback .= "Please check for infinite loops or operations that hang.`n"
            } else if (execResult.stderr != "") {
                feedback .= "AHK error output:`n" . execResult.stderr . "`n`n"
            }

            if (execResult.stdout != "" && execResult.stdout != "Script completed successfully") {
                feedback .= "Partial output before failure:`n" . execResult.stdout . "`n`n"
            }

            feedback .= "Please diagnose the issue and generate a corrected version of the script."

        } else if (attemptNum = 2) {
            ; Attempt 2: Detailed, request different approach
            feedback := "The previous fix did not resolve the issue. The script failed again.`n`n"

            if (execResult.stderr != "")
                feedback .= "Error output:`n" . execResult.stderr . "`n`n"

            feedback .= "Please generate a completely new version of the script, "
            feedback .= "taking a fundamentally different approach to the failing step. "
            feedback .= "For example, if window title matching failed, try process-based matching. "
            feedback .= "If click coordinates failed, try keyboard navigation."

        } else if (attemptNum = 3) {
            ; Attempt 3: Request decomposition
            feedback := "The script has failed twice at the same step. "
            feedback .= "Please break this workflow into smaller, independent parts "
            feedback .= "so each can be tested separately. Focus on isolating the "
            feedback .= "failing step into its own standalone operation.`n`n"

            if (execResult.stderr != "")
                feedback .= "Latest error:`n" . execResult.stderr

        } else {
            ; Attempt 4+: Last resort
            feedback := "The script continues to fail after " . attemptNum . " attempts. "
            feedback .= "Please provide a minimal, simplified version that handles only "
            feedback .= "the core operation with maximum error handling and logging.`n`n"

            if (execResult.stderr != "")
                feedback .= "Error:`n" . execResult.stderr
        }

        return feedback
    }

    ; --------------------------------------------------------------------
    ; _ToFloat -- Safely convert a value to a float
    ; --------------------------------------------------------------------

    _ToFloat(val) {
        try {
            if (val is Number)
                return val + 0.0
            return Float(String(val))
        } catch {
            return 0.0
        }
    }

    ; --------------------------------------------------------------------
    ; _CleanupTempFile -- Delete a temporary file (ignore errors)
    ; --------------------------------------------------------------------

    _CleanupTempFile(path) {
        if (path = "")
            return
        try {
            if FileExist(path)
                FileDelete(path)
        }
    }
}


; ============================================================================
; Usage Examples (for reference -- not executed when #Included)
; ============================================================================
;
; Example 1: Basic workflow generation
;
;   #Include "..\vendor\Jxon.ahk"
;   #Include "HttpClient.ahk"
;   #Include "AIEngine.ahk"
;
;   client := HttpClient("https://claude-ahk.hercules.dev")
;   client.SetAuth("cak_tk_your_token")
;
;   engine := AIEngine(client)
;
;   ; Assume workflow is a Map from WorkflowSerializer.SerializeWorkflow()
;   workflow := Map(
;       "workflow", Map(
;           "name", "Fill expense report",
;           "actions", [ ... ]
;       )
;   )
;
;   try {
;       result := engine.GenerateScript(workflow)
;       MsgBox("Generated script (confidence: " . result.confidenceScore . "):`n`n" . result.explanation)
;
;       if (result.warnings.Length > 0) {
;           warningText := ""
;           for _, w in result.warnings
;               warningText .= w . "`n"
;           MsgBox("Warnings:`n" . warningText)
;       }
;   } catch AIEngineError as err {
;       MsgBox("Generation failed: " . err.Message)
;   }
;
;
; Example 2: Execute with auto-refinement
;
;   result := engine.GenerateScript(workflow)
;   autoResult := engine.AutoExecuteWithRefinement(
;       result.scriptCode,
;       result.sessionId
;   )
;
;   if (autoResult["success"]) {
;       MsgBox("Script succeeded on attempt " . autoResult["attempts"])
;   } else {
;       MsgBox("Script failed after " . autoResult["attempts"] . " attempts")
;       for _, entry in autoResult["history"] {
;           MsgBox("Attempt " . entry["attempt"] . ": " . entry["action"])
;       }
;   }
;
;
; Example 3: Manual refinement loop
;
;   result := engine.GenerateScript(workflow)
;   execResult := engine.ExecuteScript(result.scriptCode)
;
;   if (!execResult.success) {
;       refined := engine.RefineScript(
;           result.scriptCode,
;           "The Chrome window was not found because the title changed.",
;           result.sessionId,
;           Map("error_output", execResult.stderr)
;       )
;       MsgBox("Refined script confidence: " . refined.confidenceScore)
;   }
;
;
; Example 4: Analysis only (no code generation)
;
;   analysis := engine.AnalyzeWorkflow(workflow)
;   MsgBox("Intent: " . analysis["intent"] . "`nStrategy: " . analysis["target_strategy"])
;
;
; Example 5: Save generated script to disk
;
;   result := engine.GenerateScript(workflow)
;   paths := engine.SaveScript(result.scriptCode, "ExpenseReport")
;   MsgBox("Script saved to: " . paths["scriptPath"])
;
; ============================================================================
