; ============================================================================
; Main.ahk -- Entry Point for Claude AHK (AutoHotkey v2.0.21+)
; ============================================================================
;
; Double-click to launch. Controls via system tray icon + hotkeys.
;
; Usage:
;   AutoHotkey64.exe Main.ahk [--config PATH] [--server URL] [--minimized] [--debug]
;
; ============================================================================

#Requires AutoHotkey v2.0
#SingleInstance Force
#Warn Unreachable, Off
#Warn All, Off

; ---------------------------------------------------------------------------
; Includes -- vendor libraries first, then application modules
; ---------------------------------------------------------------------------
#Include "vendor\Jxon.ahk"
#Include "Config.ahk"
#Include "lib\HttpClient.ahk"
#Include "lib\Recorder.ahk"
#Include "lib\UIACapture.ahk"
#Include "lib\Serializer.ahk"
#Include "lib\AIEngine.ahk"

; ---------------------------------------------------------------------------
; Application Constants
; ---------------------------------------------------------------------------
global APP_NAME    := "Claude AHK"
global APP_VERSION := "1.0.0"
global APP_TITLE   := APP_NAME " v" APP_VERSION

; ---------------------------------------------------------------------------
; Parse Command Line Arguments
; ---------------------------------------------------------------------------
global CLI_ConfigPath := ""
global CLI_ServerURL  := ""
global CLI_Minimized  := 0
global CLI_Debug      := 0

_ParseArgs()

; ---------------------------------------------------------------------------
; Initialize Logger (before anything else, so all modules can log)
; ---------------------------------------------------------------------------
global AppLog := Logger()
AppLog.SetLevel(CLI_Debug ? "DEBUG" : "INFO")
AppLog.Info("Main", "========================================")
AppLog.Info("Main", APP_TITLE " starting")
AppLog.Info("Main", "AHK version: " A_AhkVersion)
AppLog.Info("Main", "Working dir: " A_ScriptDir)

; ---------------------------------------------------------------------------
; Bootstrap Application
; ---------------------------------------------------------------------------
global App := ""

try {
    App := ClaudeAHK()
} catch as err {
    ; Show FULL error details so we can debug
    errDetail := "Fatal startup error:`n`n"
    errDetail .= "Message: " err.Message "`n"
    errDetail .= "What: " (err.HasProp("What") ? err.What : "N/A") "`n"
    errDetail .= "File: " (err.HasProp("File") ? err.File : "N/A") "`n"
    errDetail .= "Line: " (err.HasProp("Line") ? err.Line : "N/A") "`n"
    errDetail .= "Extra: " (err.HasProp("Extra") ? err.Extra : "N/A") "`n"
    errDetail .= "`nStack:`n" (err.HasProp("Stack") ? err.Stack : "N/A")
    if AppLog
        AppLog.Error("Main", errDetail)
    MsgBox(errDetail, APP_TITLE " - Startup Error", "Icon!")
    ExitApp(1)
}

; ---------------------------------------------------------------------------
; Keep script running (AHK v2 event loop)
; ---------------------------------------------------------------------------
Persistent()
return

; ===========================================================================
; ClaudeAHK -- Main Application Coordinator
; ===========================================================================

class ClaudeAHK {

    ; All properties initialized in __New() to avoid prototype read-only issues
    config     := ""
    httpClient := ""
    recorder   := ""
    uiaCapture := ""
    serializer := ""
    aiEngine   := ""
    version    := ""
    isRunning  := 0

    __New() {
        this.version   := APP_VERSION
        this.isRunning := 1
        this._hotkeys  := Map()

        ; ==============================================================
        ; Step 1: Load Configuration
        ; ==============================================================
        configPath := CLI_ConfigPath
        this.config := ConfigManager(configPath)
        AppLog.Info("Main", "Step 1 OK: Configuration loaded from: " this.config.filePath)

        ; Apply CLI overrides
        if (CLI_ServerURL != "")
            this.config.Set("Server", "URL", CLI_ServerURL)

        ; ==============================================================
        ; Step 2: Initialize HttpClient
        ; ==============================================================
        serverUrl := this.config.Get("Server", "URL")
        timeout   := this.config.GetInt("Server", "Timeout", 30000)
        timeoutSec := Max(1, timeout // 1000)

        this.httpClient := HttpClient(serverUrl, timeoutSec)

        ; Load auth token from config if available
        token := this.config.Get("Auth", "Token", "")
        if (token != "")
            this.httpClient.SetAuth(token)

        AppLog.Info("Main", "Step 2 OK: HttpClient initialized -> " serverUrl)

        ; ==============================================================
        ; Step 3: Initialize RecordingSession
        ; ==============================================================
        try {
            this.recorder := RecordingSession(this.config)
            AppLog.Info("Main", "Step 3 OK: RecordingSession initialized")
        } catch as recErr {
            AppLog.Warn("Main", "Step 3 WARN: RecordingSession failed: " recErr.Message " (continuing without)")
            this.recorder := ""
        }

        ; ==============================================================
        ; Step 4: Initialize UIACapture
        ; ==============================================================
        try {
            uiaConfig := this.config.GetUIACaptureConfig()
            this.uiaCapture := UIACapture(uiaConfig)
            if this.uiaCapture.IsAvailable()
                AppLog.Info("Main", "Step 4 OK: UIACapture initialized (UIA available)")
            else
                AppLog.Warn("Main", "Step 4 OK: UIACapture initialized (UIA NOT available)")
        } catch as uiaErr {
            AppLog.Warn("Main", "Step 4 WARN: UIACapture failed: " uiaErr.Message " (continuing without)")
            this.uiaCapture := ""
        }

        ; ==============================================================
        ; Step 5: Initialize WorkflowSerializer
        ; ==============================================================
        try {
            serializerConfig := this.config.GetSerializerConfig()
            this.serializer := WorkflowSerializer(serializerConfig)
            AppLog.Info("Main", "Step 5 OK: WorkflowSerializer initialized")
        } catch as serErr {
            AppLog.Warn("Main", "Step 5 WARN: WorkflowSerializer failed: " serErr.Message)
            this.serializer := ""
        }

        ; ==============================================================
        ; Step 6: Initialize AIEngine
        ; ==============================================================
        try {
            aiConfig := this.config.GetAIEngineConfig()
            this.aiEngine := AIEngine(this.httpClient, aiConfig)
            AppLog.Info("Main", "Step 6 OK: AIEngine initialized")
        } catch as aiErr {
            AppLog.Warn("Main", "Step 6 WARN: AIEngine failed: " aiErr.Message)
            this.aiEngine := ""
        }

        ; ==============================================================
        ; Step 7: Server Health Check (non-blocking)
        ; ==============================================================
        this._CheckServerHealth()

        ; ==============================================================
        ; Step 8: Register Global Hotkeys
        ; ==============================================================
        this._RegisterHotkeys()

        ; ==============================================================
        ; Step 9: Set Up Tray Menu
        ; ==============================================================
        this._SetupTrayMenu()

        ; ==============================================================
        ; Step 10: Startup Complete
        ; ==============================================================
        AppLog.Info("Main", "All modules initialized. " APP_TITLE " is ready.")
    }

    ; ====================================================================
    ; Server Health Check
    ; ====================================================================

    _CheckServerHealth() {
        try {
            response := this.httpClient.Get("/api/v1/health")
            AppLog.Info("Main", "Server health check: OK")
        } catch as err {
            AppLog.Warn("Main", "Server unreachable: " err.Message " (will retry on demand)")
        }
    }

    ; ====================================================================
    ; Hotkey Registration
    ; ====================================================================

    _RegisterHotkeys() {
        hkStart    := this.config.Get("Hotkeys", "StartRecording", "F5")
        hkStop     := this.config.Get("Hotkeys", "StopRecording", "F6")
        hkPause    := this.config.Get("Hotkeys", "PauseRecording", "F7")
        hkGenerate := this.config.Get("Hotkeys", "GenerateScript", "^g")
        hkExecute  := this.config.Get("Hotkeys", "ExecuteScript", "^e")
        hkEmStop   := this.config.Get("Hotkeys", "EmergencyStop", "^+x")

        this._SafeHotkey(hkStart,    (*) => this.OnStartRecording(),   "StartRecording")
        this._SafeHotkey(hkStop,     (*) => this.OnStopRecording(),    "StopRecording")
        this._SafeHotkey(hkPause,    (*) => this.OnPauseRecording(),   "PauseRecording")
        this._SafeHotkey(hkGenerate, (*) => this.OnGenerateScript(),   "GenerateScript")
        this._SafeHotkey(hkExecute,  (*) => this.OnExecuteScript(),    "ExecuteScript")
        this._SafeHotkey(hkEmStop,   (*) => this.OnEmergencyStop(),    "EmergencyStop")

        AppLog.Info("Main", "Hotkeys registered: F5=Start F6=Stop F7=Pause Ctrl+G=Generate Ctrl+E=Execute Ctrl+Shift+X=EmStop")
    }

    _SafeHotkey(hotkeyStr, callback, name) {
        try {
            Hotkey(hotkeyStr, callback)
            this._hotkeys[name] := hotkeyStr
        } catch as err {
            AppLog.Warn("Main", "Failed to register hotkey '" name "' (" hotkeyStr "): " err.Message)
        }
    }

    ; ====================================================================
    ; Hotkey Handlers
    ; ====================================================================

    OnStartRecording() {
        if (!this.recorder)
            return
        if (this.recorder.state = "idle" || this.recorder.state = "paused") {
            this.recorder.Start()
            AppLog.Info("Main", "Recording started")
            ToolTip("Recording...")
            SetTimer((*) => ToolTip(), -2000)
        }
    }

    OnStopRecording() {
        if (!this.recorder)
            return
        if (this.recorder.state = "recording" || this.recorder.state = "paused") {
            events := this.recorder.Stop()
            AppLog.Info("Main", "Recording stopped. Captured " events.Length " events")
            ToolTip("Stopped. " events.Length " events captured.")
            SetTimer((*) => ToolTip(), -3000)
        }
    }

    OnPauseRecording() {
        if (!this.recorder)
            return
        if (this.recorder.state = "recording") {
            this.recorder.Pause()
            AppLog.Info("Main", "Recording paused")
            ToolTip("Paused")
            SetTimer((*) => ToolTip(), -2000)
        } else if (this.recorder.state = "paused") {
            this.recorder.Start()
            AppLog.Info("Main", "Recording resumed")
            ToolTip("Resumed")
            SetTimer((*) => ToolTip(), -2000)
        }
    }

    OnGenerateScript() {
        if (!this.recorder || !this.serializer || !this.aiEngine)
            return

        events := this.recorder.GetEvents()
        if (!events || events.Length = 0) {
            ToolTip("No recorded events to generate from")
            SetTimer((*) => ToolTip(), -3000)
            return
        }

        ToolTip("Generating script from " events.Length " events...")
        AppLog.Info("Main", "Generating script from " events.Length " events...")

        try {
            workflow := this.serializer.SerializeWorkflow(events)
            result := this.aiEngine.GenerateScript(workflow)
            if (IsObject(result) && result.HasProp("scriptCode") && result.scriptCode != "") {
                confidence := result.HasProp("confidenceScore") ? result.confidenceScore : "N/A"
                AppLog.Info("Main", "Script generated (confidence: " confidence ")")
                ToolTip("Script generated! Ctrl+E to execute.")
                SetTimer((*) => ToolTip(), -4000)
            } else {
                ToolTip("Script generation returned empty")
                SetTimer((*) => ToolTip(), -3000)
            }
        } catch as err {
            AppLog.Error("Main", "Script generation failed: " err.Message)
            ToolTip("Generation failed: " err.Message)
            SetTimer((*) => ToolTip(), -5000)
        }
    }

    OnExecuteScript() {
        if (!this.aiEngine)
            return

        lastRes := this.aiEngine.lastResult
        if (!IsObject(lastRes) || !lastRes.HasProp("scriptCode") || lastRes.scriptCode = "") {
            ToolTip("No script to execute. Record + Ctrl+G first.")
            SetTimer((*) => ToolTip(), -3000)
            return
        }

        ToolTip("Executing script...")
        AppLog.Info("Main", "Executing last generated script...")

        try {
            result := this.aiEngine.ExecuteScript(lastRes.scriptCode)
            if IsObject(result) {
                success := result.HasProp("success") ? result.success : 0
                if success {
                    AppLog.Info("Main", "Script executed successfully")
                    ToolTip("Script executed OK")
                } else {
                    AppLog.Warn("Main", "Script execution reported failure")
                    ToolTip("Script execution failed")
                }
                SetTimer((*) => ToolTip(), -3000)
            }
        } catch as err {
            AppLog.Error("Main", "Script execution failed: " err.Message)
            ToolTip("Execution error: " err.Message)
            SetTimer((*) => ToolTip(), -5000)
        }
    }

    OnEmergencyStop() {
        AppLog.Warn("Main", "EMERGENCY STOP triggered")
        ToolTip("EMERGENCY STOP")
        SetTimer((*) => ToolTip(), -2000)

        if (this.recorder && this.recorder.state != "idle")
            try this.recorder.Stop()

        if (this.aiEngine)
            try this.aiEngine.StopExecution()

        AppLog.Info("Main", "All operations halted")
    }

    ; ====================================================================
    ; Tray Menu
    ; ====================================================================

    _SetupTrayMenu() {
        tray := A_TrayMenu
        tray.Delete()

        tray.Add("Show Window", (*) => this._ShowWindow())
        tray.Default := "Show Window"
        tray.Add()
        tray.Add("Start Recording (F5)", (*) => this.OnStartRecording())
        tray.Add("Stop Recording (F6)",  (*) => this.OnStopRecording())
        tray.Add()
        tray.Add("About " APP_NAME, (*) => this._ShowAbout())
        tray.Add()
        tray.Add("Exit",            (*) => this.Shutdown())

        A_IconTip := APP_TITLE
        AppLog.Debug("Main", "Tray menu configured")
    }

    ; ====================================================================
    ; Window Management
    ; ====================================================================

    _ShowWindow() {
        MsgBox(APP_TITLE "`n`nHotkeys:`n"
            . "  F5 - Start Recording`n"
            . "  F6 - Stop Recording`n"
            . "  F7 - Pause/Resume`n"
            . "  Ctrl+G - Generate Script`n"
            . "  Ctrl+E - Execute Script`n"
            . "  Ctrl+Shift+X - Emergency Stop`n`n"
            . "Config: " this.config.filePath,
            APP_TITLE, "Iconi")
    }

    _ShowAbout() {
        MsgBox(
            APP_NAME " - AI Desktop Automation`n"
            . "Version: " APP_VERSION "`n`n"
            . "Powered by Claude on Hercules`n`n"
            . "Record your desktop workflows and let AI`n"
            . "generate AutoHotkey v2 scripts to automate them.",
            "About " APP_NAME, "Iconi"
        )
    }

    ; ====================================================================
    ; Lifecycle
    ; ====================================================================

    Shutdown() {
        AppLog.Info("Main", "Shutdown requested")
        this.isRunning := 0

        if (this.recorder && this.recorder.state != "idle")
            try this.recorder.Stop()

        if (this.aiEngine)
            try this.aiEngine.StopExecution()

        try this.config.Save()

        for name, hkStr in this._hotkeys
            try Hotkey(hkStr, "Off")

        AppLog.Info("Main", APP_TITLE " shutdown complete")
        ExitApp(0)
    }
}

; ===========================================================================
; Logger -- Simple file-based logger with rotation
; ===========================================================================

class Logger {

    filePath   := ""
    maxSize    := 5242880
    level      := "INFO"

    __New() {
        this._levels := Map("DEBUG", 1, "INFO", 2, "WARN", 3, "ERROR", 4)
        logDir := A_ScriptDir "\logs"
        if !DirExist(logDir)
            try DirCreate(logDir)
        this.filePath := logDir "\claude-ahk.log"
    }

    SetLevel(lvl) {
        this.level := StrUpper(lvl)
    }

    WriteLog(level, source, message) {
        level := StrUpper(level)
        if !this._levels.Has(level)
            return
        if !this._levels.Has(this.level)
            return
        if (this._levels[level] < this._levels[this.level])
            return

        timestamp := FormatTime(, "yyyy-MM-dd HH:mm:ss")
        line := "[" timestamp "] [" level "] [" source "] " message "`n"

        this._RotateIfNeeded()
        try FileAppend(line, this.filePath)
    }

    Debug(source, message) => this.WriteLog("DEBUG", source, message)
    Info(source, message)  => this.WriteLog("INFO", source, message)
    Warn(source, message)  => this.WriteLog("WARN", source, message)
    Error(source, message) => this.WriteLog("ERROR", source, message)

    _RotateIfNeeded() {
        if !FileExist(this.filePath)
            return
        try {
            fileObj := FileOpen(this.filePath, "r")
            if !fileObj
                return
            size := fileObj.Length
            fileObj.Close()
            if (size > this.maxSize) {
                bakPath := this.filePath ".bak"
                if FileExist(bakPath)
                    FileDelete(bakPath)
                FileMove(this.filePath, bakPath)
            }
        }
    }
}

; ===========================================================================
; Command Line Argument Parser
; ===========================================================================

_ParseArgs() {
    global CLI_ConfigPath, CLI_ServerURL, CLI_Minimized, CLI_Debug
    args := A_Args
    i := 1
    while (i <= args.Length) {
        arg := args[i]

        if (arg = "--config" && i < args.Length) {
            i++
            CLI_ConfigPath := args[i]
        }
        else if (arg = "--server" && i < args.Length) {
            i++
            CLI_ServerURL := args[i]
        }
        else if (arg = "--minimized") {
            CLI_Minimized := 1
        }
        else if (arg = "--debug") {
            CLI_Debug := 1
        }

        i++
    }
}
