; ============================================================================
; Config.ahk -- Configuration Management for Claude AHK (AutoHotkey v2.0.21+)
; ============================================================================
;
; Manages persistent application settings using INI files. Provides typed
; access to configuration values with automatic default creation on first run.
;
; Module:   Config.ahk (root level, alongside Main.ahk)
; Version:  1.0.0
;
; Classes:
;   ConfigManager  -- INI-backed configuration with defaults and typed accessors
;
; Usage:
;   cfg := ConfigManager()                     ; uses default claude-ahk.ini
;   cfg := ConfigManager("C:\path\to\my.ini")  ; custom path
;
;   url := cfg.Get("Server", "URL")
;   timeout := cfg.GetInt("Server", "Timeout")
;   alwaysOnTop := cfg.GetBool("GUI", "AlwaysOnTop")
;
;   cfg.Set("Server", "URL", "https://example.com")
;   cfg.Save()
;
; ============================================================================

#Requires AutoHotkey v2.0

class ConfigManager {

    ; --- Properties ---
    filePath := ""        ; Absolute path to the INI file
    data     := Map()     ; Nested Map: section -> Map(key -> value)
    defaults := Map()     ; Nested Map: section -> Map(key -> value)

    ; --- Constructor ---
    ; configPath : String -- Path to the INI file. Empty = A_ScriptDir "\claude-ahk.ini"
    __New(configPath := "") {
        if (configPath = "")
            this.filePath := A_ScriptDir "\claude-ahk.ini"
        else
            this.filePath := configPath

        this._PopulateDefaults()
        this.Load()
    }

    ; ========================================================================
    ; Public API
    ; ========================================================================

    ; Load -- Read all settings from INI file.
    ; Creates the INI file with defaults if it does not exist.
    Load() {
        this.data := Map()

        ; If file does not exist, create it with all defaults
        if !FileExist(this.filePath)
            this._CreateDefaultFile()

        ; Read each section/key pair from INI, falling back to the default
        for section, keys in this.defaults {
            sectionMap := Map()
            for key, defaultVal in keys {
                value := IniRead(this.filePath, section, key, defaultVal)
                sectionMap[key] := value
            }
            this.data[section] := sectionMap
        }
    }

    ; Save -- Write all current settings to INI file.
    ; Skips sensitive Auth token fields (those are managed via CredManager).
    Save() {
        for section, keys in this.data {
            for key, value in keys {
                ; Skip writing auth tokens to INI -- they belong in CredManager
                if (section = "Auth" && (key = "Token" || key = "RefreshToken"))
                    continue
                IniWrite(value, this.filePath, section, key)
            }
        }
    }

    ; Get -- Retrieve a setting value as a string.
    ; section : String -- INI section name (e.g. "Server")
    ; key     : String -- Setting key (e.g. "URL")
    ; default : String -- Fallback if section/key not found in data or defaults
    ; Returns : String
    Get(section, key, default := "") {
        if this.data.Has(section) && this.data[section].Has(key)
            return this.data[section][key]
        if this.defaults.Has(section) && this.defaults[section].Has(key)
            return this.defaults[section][key]
        return default
    }

    ; GetInt -- Retrieve a setting value as an integer.
    ; Falls back to 0 if the value cannot be parsed.
    GetInt(section, key, default := 0) {
        raw := this.Get(section, key, "")
        if (raw = "")
            return default
        try
            return Integer(raw)
        catch
            return default
    }

    ; GetFloat -- Retrieve a setting value as a float.
    GetFloat(section, key, default := 0.0) {
        raw := this.Get(section, key, "")
        if (raw = "")
            return default
        try
            return Float(raw)
        catch
            return default
    }

    ; GetBool -- Retrieve a setting value as a boolean (1 or 0).
    ; Recognizes: 1, true, yes, on -> 1 ; 0, false, no, off -> 0
    GetBool(section, key, default := 0) {
        raw := this.Get(section, key, "")
        if (raw = "")
            return default
        lower := StrLower(String(raw))
        if (lower = "1" || lower = "true" || lower = "yes" || lower = "on")
            return 1
        if (lower = "0" || lower = "false" || lower = "no" || lower = "off")
            return 0
        return default
    }

    ; Set -- Set a config value in memory and persist to INI.
    ; section : String
    ; key     : String
    ; value   : Any -- will be converted to string for storage
    Set(section, key, value) {
        if !this.data.Has(section)
            this.data[section] := Map()
        this.data[section][key] := String(value)

        ; Persist immediately (skip auth tokens)
        if !(section = "Auth" && (key = "Token" || key = "RefreshToken"))
            IniWrite(String(value), this.filePath, section, key)
    }

    ; GetAll -- Return all settings as a nested Map.
    ; Returns a deep copy so callers cannot accidentally mutate internal state.
    GetAll() {
        result := Map()
        for section, keys in this.data {
            sectionCopy := Map()
            for key, value in keys
                sectionCopy[key] := value
            result[section] := sectionCopy
        }
        return result
    }

    ; GetSection -- Return all keys in a section as a Map.
    ; Returns empty Map if section does not exist.
    GetSection(section) {
        result := Map()
        if this.data.Has(section) {
            for key, value in this.data[section]
                result[key] := value
        }
        return result
    }

    ; ResetToDefaults -- Replace all settings with defaults and rewrite INI.
    ResetToDefaults() {
        this.data := Map()
        for section, keys in this.defaults {
            sectionMap := Map()
            for key, value in keys
                sectionMap[key] := value
            this.data[section] := sectionMap
        }

        ; Delete existing file and recreate
        if FileExist(this.filePath) {
            try FileDelete(this.filePath)
        }
        this._CreateDefaultFile()
    }

    ; HasSection -- Check if a section exists in loaded data.
    HasSection(section) {
        return this.data.Has(section)
    }

    ; HasKey -- Check if a section/key pair exists in loaded data.
    HasKey(section, key) {
        return this.data.Has(section) && this.data[section].Has(key)
    }

    ; ========================================================================
    ; Internal Methods
    ; ========================================================================

    ; _PopulateDefaults -- Define all default configuration values.
    ; This is the single source of truth for all supported settings.
    _PopulateDefaults() {
        d := Map()

        ; --- Server ---
        d["Server"] := Map(
            "URL",      "https://claude-ahk.herakles.dev",
            "Port",     "8154",
            "Timeout",  "30000"
        )

        ; --- Auth ---
        d["Auth"] := Map(
            "Token",       "",
            "RefreshToken", "",
            "TokenExpiry",  ""
        )

        ; --- Recording ---
        d["Recording"] := Map(
            "MousePollInterval",  "50",
            "KeyboardBufferSize", "50000",
            "MaxRecordDuration",  "300000",
            "IncludeMouseMove",   "1"
        )

        ; --- Serializer ---
        d["Serializer"] := Map(
            "KeyGroupThreshold",  "500",
            "MouseMergeThreshold", "200",
            "ClickDedupeRadius",   "3",
            "RDPEpsilon",          "5.0",
            "MaxActions",          "1000"
        )

        ; --- UICapture ---
        d["UICapture"] := Map(
            "MaxTreeDepth",    "5",
            "MaxElements",     "500",
            "CaptureTimeout",  "3000",
            "AutoCapture",     "1"
        )

        ; --- AI ---
        d["AI"] := Map(
            "ExecutionTimeout",     "60",
            "MaxRetries",           "4",
            "AutoRefine",           "0",
            "ConfidenceThreshold",  "0.7"
        )

        ; --- GUI ---
        d["GUI"] := Map(
            "Width",       "900",
            "Height",      "700",
            "FontSize",    "10",
            "CodeFont",    "Consolas",
            "Theme",       "dark",
            "AlwaysOnTop", "0"
        )

        ; --- Hotkeys ---
        d["Hotkeys"] := Map(
            "StartRecording",  "F5",
            "StopRecording",   "F6",
            "PauseRecording",  "F7",
            "GenerateScript",  "^g",
            "ExecuteScript",   "^e",
            "EmergencyStop",   "^+x"
        )

        ; --- Paths ---
        d["Paths"] := Map(
            "ScriptSaveDir", "scripts",
            "LogDir",        "logs",
            "AhkExePath",    "AutoHotkey64.exe"
        )

        this.defaults := d
    }

    ; _CreateDefaultFile -- Write a fresh INI file with all default values.
    _CreateDefaultFile() {
        ; Ensure parent directory exists
        SplitPath(this.filePath, , &dir)
        if (dir != "" && !DirExist(dir)) {
            try DirCreate(dir)
        }

        ; Write every default section/key to the INI file
        for section, keys in this.defaults {
            for key, value in keys {
                ; Skip writing empty auth tokens
                if (section = "Auth" && value = "")
                    continue
                IniWrite(value, this.filePath, section, key)
            }
        }
    }

    ; --- Convenience: build a config Map for RecordingSession ---
    ; RecordingSession expects a Config-like object with GetInt() method,
    ; so we pass `this` directly. These methods exist for that purpose.

    ; --- Convenience: build module-specific config Maps ---

    ; GetRecordingConfig -- Returns this ConfigManager (RecordingSession uses GetInt)
    GetRecordingConfig() {
        return this
    }

    ; GetUIACaptureConfig -- Returns a Map suitable for UIACapture constructor.
    GetUIACaptureConfig() {
        return Map(
            "maxDepth",    this.GetInt("UICapture", "MaxTreeDepth", 5),
            "maxElements", this.GetInt("UICapture", "MaxElements", 500),
            "timeout",     this.GetInt("UICapture", "CaptureTimeout", 3000),
            "enabled",     this.GetBool("UICapture", "AutoCapture", 1)
        )
    }

    ; GetSerializerConfig -- Returns a Map suitable for WorkflowSerializer constructor.
    GetSerializerConfig() {
        return Map(
            "keyGroupThreshold",   this.GetInt("Serializer", "KeyGroupThreshold", 500),
            "mouseMergeThreshold", this.GetInt("Serializer", "MouseMergeThreshold", 200),
            "clickDedupeRadius",   this.GetInt("Serializer", "ClickDedupeRadius", 3),
            "rdpEpsilon",          this.GetFloat("Serializer", "RDPEpsilon", 5.0),
            "maxActions",          this.GetInt("Serializer", "MaxActions", 1000)
        )
    }

    ; GetAIEngineConfig -- Returns a Map suitable for AIEngine constructor.
    GetAIEngineConfig() {
        return Map(
            "executionTimeout", this.GetInt("AI", "ExecutionTimeout", 60),
            "maxRetries",       this.GetInt("AI", "MaxRetries", 4),
            "autoRefine",       this.GetBool("AI", "AutoRefine", 0),
            "scriptSaveDir",    this._ResolvePath(this.Get("Paths", "ScriptSaveDir", "scripts")),
            "ahkExePath",       this._ResolvePath(this.Get("Paths", "AhkExePath", "AutoHotkey64.exe"))
        )
    }

    ; _ResolvePath -- Resolve a relative path against A_ScriptDir.
    ; If path is already absolute (contains : or starts with \\), return as-is.
    _ResolvePath(path) {
        if (InStr(path, ":") || SubStr(path, 1, 2) = "\\")
            return path
        return A_ScriptDir "\" path
    }
}
